#include "client_filter_helpers.h"

#include <infra/libs/controller/controller/error/whitelist_error.h>

#include <yt/yt/orm/library/filter_introspection.h>

#include <yt/yt/client/table_client/row_buffer.h>

#include <yt/yt/core/ypath/tokenizer.h>
#include <yt/yt/core/yson/string.h>
#include <yt/yt/core/ytree/fluent.h>

#include <util/string/reverse.h>
#include <util/string/split.h>

using namespace NInfra::NController;

struct TRowBufferTag
{ };

TStringBuf GetClusterNameByClient(
    const IObjectManager::TSelectArgument& selectArgument,
    NYP::NClient::TClientPtr client
) {
    if (!selectArgument.ClusterName || selectArgument.ClusterName->empty()) {
        return TStringBuf(client->Options().Address()).Before('.').Before(':');
    }
    return *selectArgument.ClusterName;
}

void ValidateFilter(const TString& filter, const TSet<TString>& whitelist) {
    TString currentPath;
    bool openedPath = false;
    bool screen = false;
    for (const char symbol : filter) {
        if (symbol == ']' && !screen) {
            if (!whitelist.contains(currentPath)) {
                throw TWhitelistError() << "Selector [" << currentPath << "] in filter is not in whitelist.";
            }

            currentPath.clear();
            openedPath = false;
        } else if (symbol == '[' && !screen) {
            openedPath = true;
        } else if (openedPath) {
            currentPath += symbol;
        }

        if (screen) {
            screen = false;
        } else if (symbol == '\\') {
            screen = true;
        }
    }
}

TVector<TString> ExtractClientFilterSelectorsFromArgument(
    const IObjectManager::TSelectArgument& selectArg
    , const TStringBuf cluster
) {
    TVector<TString> result;

    auto addSelector = [&result](const TString& selector) {
        if (result.empty()) {
            result.emplace_back(selector);
        } else if (result.back() != selector && (!TStringBuf(selector).SkipPrefix(result.back()) || selector[result.back().size()] != '/')) {
            result.emplace_back(selector);
        }
    };

    if (!selectArg.ClientFilterOptions.Enabled) {
        return {};
    }

    const auto& whitelistData = selectArg.ClientFilterOptions.GetWhitelistData(cluster);

    if (!whitelistData.Defined()) {
        ythrow yexception() << "Whitelist for cluster is not defined, but client filter is enabled.";
    }

    ValidateFilter(selectArg.Filter, whitelistData->Selectors);

    for (const auto& field : whitelistData->Selectors) {
        if (selectArg.ClientFilterOptions.MergeLabels && NYT::NYPath::HasPrefix(field, "/labels")) {
            addSelector("/labels");
        } else {
            addSelector(field);
        }
    }

    return result;
}

TVector<TString> ExtractAllSelectors(
    const IObjectManager::TSelectArgument& selectArgument
    , const TStringBuf cluster
) {
    TVector<TString> clientFilterSelectors = ExtractClientFilterSelectorsFromArgument(selectArgument, cluster);
    TVector<TString> selectors = selectArgument.Selector;
    selectors.reserve(clientFilterSelectors.size() + selectors.size());
    selectors.insert(selectors.end(), clientFilterSelectors.begin(), clientFilterSelectors.end());
    return selectors;
}

TVector<size_t> GetIndependentAttributesIndices(const TVector<TString>& selectors) {
    TVector<size_t> sortedSelectors(selectors.size()); 
    Iota(sortedSelectors.begin(), sortedSelectors.end(), 0);
    SortBy(sortedSelectors, [&selectors](const size_t& i) {
        return selectors[i];
    });

    TVector<size_t> result;
    if (!sortedSelectors.empty()) {
        result.emplace_back(sortedSelectors[0]);
    }

    for (size_t i = 0; i < sortedSelectors.size(); ++i) {
        if (selectors[sortedSelectors[i]] != selectors[result.back()] && 
            (!NYT::NYPath::HasPrefix(selectors[sortedSelectors[i]], selectors[result.back()]) || selectors[sortedSelectors[i]][selectors[result.back()].size()] != '/')) {
            result.emplace_back(sortedSelectors[i]);
        }
    }

    return result;
}

TCachedFilterMatcher CreateFilterMatcher(
    const IObjectManager::TSelectArgument& selectArgument,
    const TMaybe<TString>& extractedFilter,
    NYP::NClient::TClientPtr client
) {
    const TStringBuf cluster = GetClusterNameByClient(selectArgument, client);
    if (!selectArgument.ClientFilterOptions.Enabled) {
        return TCachedFilterMatcher(NYT::NOrm::NLibrary::CreateConstantFilterMatcher(true), TVector<size_t>());
    }

    const auto& whitelistData = selectArgument.ClientFilterOptions.GetWhitelistData(cluster);
    if (!whitelistData.Defined()) {
        ythrow yexception() << "Whitelist for cluster is not defined, but client filter is enabled.";
    }

    TVector<TString> selectors = ExtractAllSelectors(selectArgument, cluster);
    auto independentAttributesIndices = GetIndependentAttributesIndices(selectors);
    TVector<TString> independentSelectors;
    independentSelectors.reserve(selectors.size());
    for (auto i : independentAttributesIndices) {
        independentSelectors.emplace_back(selectors[i]);
    }

    return TCachedFilterMatcher(
        NYT::NOrm::NLibrary::CreateFilterMatcher(
            extractedFilter.Defined() ? extractedFilter.GetRef() : selectArgument.Filter
            , std::move(independentSelectors)
        ),
        std::move(independentAttributesIndices)
    );
}

void ApplyFilterOnSelectResult(
    TSelectObjectsResult& objectsSelectResults
    , const IObjectManager::TSelectArgument& selectArgument
    , const TCachedFilterMatcher& matcher
) {
    if (!objectsSelectResults.Results.size() || !selectArgument.ClientFilterOptions.Enabled) {
        return;
    }

    auto rowBuffer = NYT::New<NYT::NTableClient::TRowBuffer>(TRowBufferTag());

    TVector<TSelectorResultPtr> filteredResults;
    for (auto& objectSelectResult : objectsSelectResults.Results) {
        std::vector<NYT::NYson::TYsonStringBuf> attributeYsons;
        attributeYsons.reserve(matcher.AttributeIndices.size());
        for (size_t i : matcher.AttributeIndices) {
            if (objectSelectResult->Values().value_payloads()[i].null() ||
                NYP::NClient::NYsonUtil::IsNull(
                    objectSelectResult->Values().value_payloads()[i].yson())
            ) {
                attributeYsons.emplace_back(NYT::NYTree::BuildYsonStringFluently().Entity());
            } else {
                attributeYsons.emplace_back(objectSelectResult->Values().value_payloads()[i].yson());
            }
        }

        if (matcher.Matcher->Match(attributeYsons, rowBuffer).ValueOrThrow()) {
            NYP::NClient::TSelectorResult enlargedObject = *objectSelectResult;
            enlargedObject.Values().mutable_value_payloads()->Truncate(selectArgument.Selector.size());
            filteredResults.emplace_back(MakeAtomicShared<NYP::NClient::TSelectorResult>(std::move(enlargedObject)));
        }
    }

    objectsSelectResults.Results = std::move(filteredResults);
}

TVector<TString> ExtractKeyFieldValue(const IObjectManager::TSelectArgument& selectArgument, const TStringBuf clusterName) {
    if (!selectArgument.ClientFilterOptions.Enabled) {
        return {};
    }

    const auto& whitelistData = selectArgument.ClientFilterOptions.GetWhitelistData(clusterName);
    if (!whitelistData.Defined()) {
        return {};
    }
    const auto& keyField = whitelistData->KeyField;

    const TString& filter = selectArgument.Filter;
    try {
        if (const auto result = NYT::NOrm::NLibrary::IntrospectFilterForDefinedAttributeValue(filter, keyField.GetRef()); result.has_value()) {
            return {result.value()};
        }
    } catch (...) {
        return {};
    }

    return {};
}
