#include "controller.h"

#include <infra/libs/controller/controller/error/whitelist_error.h>

#include <infra/libs/sensors/macros.h>
#include <yp/cpp/yp/object_traits.h>

#include <library/cpp/json/yson/json2yson.h>
#include <library/cpp/retry/retry.h>
#include <library/cpp/threading/async_task_batch/async_task_batch.h>

#include <yp/client/api/misc/public.h>

#include <util/datetime/cputimer.h>
#include <util/string/strip.h>

namespace {

TString MergeFilters(const TString& mainFilter, const TString& additionalFilter) {
    if (mainFilter.empty()) {
        return additionalFilter;
    }

    if (additionalFilter.empty()) {
        return mainFilter;
    }

    return TStringBuilder{} << "(" << mainFilter << ") and (" << additionalFilter << ")";
}

} // anonymous namespace

namespace NInfra::NController {

void TControllerBase::Sync(
    const THashMap<TString, NYP::NClient::TClientPtr>& clients
    , const THashMap<TString, NYP::NClient::TTransactionFactoryPtr>& transactionFactories
    , TThreadPool& mtpQueue
    , IThreadPool& auxMtpQueue
    , TLogFramePtr frame
    , ui32 retryCount
    , const ui32 ypRequestsBatchSize
    , const ui32 maxSeqFailedObjMngrsCount
) {
    LeaderSyncDurationSensor_.Start();
    SyncDurationSensor_.Start();

    {
        TVector<TString> currentKeys;
        currentKeys.reserve(FilterMatchers_.size());
        for (auto it : FilterMatchers_) {
            currentKeys.emplace_back(it.first);
        }

        for (auto it : currentKeys) {
            if (!UsedFilterMatchers_.contains(it)) {
                FilterMatchers_.erase(it);
            }
        }
    }

    UsedFilterMatchers_.clear();

    AtomicSet(NeedAbort_, 0);

    const TVector<TObjectManagerPtr> objectManagers = GetFactoryObjectManagers(clients, mtpQueue, frame);

    if (ShouldAbortTaskDueToMasterDistribution(frame)) {
        DestroyLeadingInvader();
    }

    auto shard = ObjectManagerFactory_->GetShard();
    TVector<NThreading::TFuture<void>> objectManagersFuture(objectManagers.size());
    for (size_t i = 0; i < objectManagers.size(); ++i) {
        TObjectManagerPtr objectManager = objectManagers[i];

        auto sync = [this, shard, objectManager, &clients, &transactionFactories, frame, ypRequestsBatchSize, retryCount, &auxMtpQueue]() {
            if (AtomicGet(NeedAbort_) == 1) {
                frame->LogEvent(
                    ELogPriority::TLOG_DEBUG
                    , NLogEvent::TSyncAborted(objectManager->GetObjectId())
                );
                return;
            }

            auto leadingResult = shard->EnsureLeading();
            if ((bool)leadingResult) {
                objectManager->SetLeadership(true);
            } else if (!IsResponsibleForLock()) {
                LeaderSyncDurationSensor_.Reset();
                ythrow yexception() << "Lost lock '" << leadingResult.Error().Reason << "'";
            }

            auto loadAllDependentObjectDataToRetry = [this, frame, &clients, objectManager, &auxMtpQueue]() {
                return LoadAllDependentObjectData(clients, objectManager, auxMtpQueue, frame);
            };

            std::function<void(const yexception&)> onFail = [this, frame, &clients, objectManager](const yexception& e) {
                NLogEvent::TSyncTryFailed event;
                event.SetObjectId(objectManager->GetObjectId());
                event.SetMessage(e.what());
                frame->LogEvent(ELogPriority::TLOG_ERR, event);

                IncrementFactorySensor("sync_object_retries", 1);

                /* If YP masters are dead - object managers will fail requests
                 * with full timeout, this takes long time.
                 * So rediscovering the masters helps to avoid this.
                 */
                for (const auto& [_, client] : clients) {
                    client->ReconstructBalancing(
                        {}
                        , MakeHolder<TSensorContext>(
                            frame
                            , HistogramSensors_.at(RECONSTRUCT_BALANCING)
                        )
                    );
                }
            };

            SyncObject(
                objectManager
                , *DoWithRetry<IObjectManager::TDependentObjects>(loadAllDependentObjectDataToRetry, onFail, TRetryOptions().WithCount(retryCount), /* throwLast */ true) // TODO(wrg0ababd): temporary solution, redo it later
                , transactionFactories
                , clients
                , ypRequestsBatchSize
                , auxMtpQueue
                , frame
                , retryCount
            );
        };

        objectManagersFuture[i] = NThreading::Async(
            sync
            , mtpQueue
        );
    }

    TMap<TString, google::protobuf::RepeatedPtrField<TString>> syncErrors; // map<message, obj_ids>
    ui32 seqFailsCount = 0;

    for (size_t i = 0; i < objectManagersFuture.size(); ++i) {
        try {
            objectManagersFuture[i].GetValueSync();
            seqFailsCount = 0;
        } catch (...) {
            ++seqFailsCount;
            if (seqFailsCount == maxSeqFailedObjMngrsCount) {
                AtomicSet(NeedAbort_, 1);
            }

            *syncErrors[CurrentExceptionMessage()].Add() = objectManagers[i]->GetObjectId();
            IncrementFactorySensor("sync_object_errors", 1);
        }
    }

    if (ShouldAbortTaskDueToMasterDistribution(frame)) {
        DestroyLeadingInvader();
    }

    if (EnsureLeading()) {
        LeaderSyncDurationSensor_.Update();
        SyncDurationSensor_.Update();
    } else {
        LeaderSyncDurationSensor_.Reset();
        if (IsResponsibleForLock()) {
            SyncDurationSensor_.Update();
        } else {
            SyncDurationSensor_.Reset();
        }
    }

    for (auto&& [message, ids] : syncErrors) {
        NLogEvent::TObjectsSyncError objectsSyncError;
        objectsSyncError.SetMessage(message);
        objectsSyncError.MutableIds()->CopyFrom(ids);

        frame->LogEvent(ELogPriority::TLOG_ERR, objectsSyncError);
    }

    if (!syncErrors.empty()) {
        // Errors' messages logged in loop above, throw any error message
        throw yexception() << syncErrors.begin()->first;
    }
}

NLogEvent::EYPObjectType TControllerBase::ConvertYPObjectType(NYP::NClient::NApi::NProto::EObjectType objectType) {
    switch (objectType) {
        case NYP::NClient::NApi::NProto::EObjectType::OT_ENDPOINT:
            return NLogEvent::OT_ENDPOINT;
        case NYP::NClient::NApi::NProto::EObjectType::OT_ENDPOINT_SET:
            return NLogEvent::OT_ENDPOINT_SET;
        case NYP::NClient::NApi::NProto::EObjectType::OT_INTERNET_ADDRESS:
            return NLogEvent::OT_INTERNET_ADDRESS;
        case NYP::NClient::NApi::NProto::EObjectType::OT_NODE:
            return NLogEvent::OT_NODE;
        case NYP::NClient::NApi::NProto::EObjectType::OT_NODE2:
            return NLogEvent::OT_NODE2;
        case NYP::NClient::NApi::NProto::EObjectType::OT_POD:
            return NLogEvent::OT_POD;
        case NYP::NClient::NApi::NProto::EObjectType::OT_RESOURCE_CACHE:
            return NLogEvent::OT_RESOURCE_CACHE;
        case NYP::NClient::NApi::NProto::EObjectType::OT_SCHEMA:
            return NLogEvent::OT_SCHEMA;
        case NYP::NClient::NApi::NProto::EObjectType::OT_POD_SET:
            return NLogEvent::OT_POD_SET;
        case NYP::NClient::NApi::NProto::EObjectType::OT_USER:
            return NLogEvent::OT_USER;
        case NYP::NClient::NApi::NProto::EObjectType::OT_GROUP:
            return NLogEvent::OT_GROUP;
        case NYP::NClient::NApi::NProto::EObjectType::OT_ACCOUNT:
            return NLogEvent::OT_ACCOUNT;
        case NYP::NClient::NApi::NProto::EObjectType::OT_DNS_RECORD_SET:
            return NLogEvent::OT_DNS_RECORD_SET;
        case NYP::NClient::NApi::NProto::EObjectType::OT_NETWORK_PROJECT:
            return NLogEvent::OT_NETWORK_PROJECT;
        case NYP::NClient::NApi::NProto::EObjectType::OT_REPLICA_SET:
            return NLogEvent::OT_REPLICA_SET;
        case NYP::NClient::NApi::NProto::EObjectType::OT_MULTI_CLUSTER_REPLICA_SET:
            return NLogEvent::OT_MULTI_CLUSTER_REPLICA_SET;
        case NYP::NClient::NApi::NProto::EObjectType::OT_STAGE:
            return NLogEvent::OT_STAGE;
        case NYP::NClient::NApi::NProto::EObjectType::OT_VIRTUAL_SERVICE:
            return NLogEvent::OT_VIRTUAL_SERVICE;
        case NYP::NClient::NApi::NProto::EObjectType::OT_NODE_SEGMENT:
            return NLogEvent::OT_NODE_SEGMENT;
        case NYP::NClient::NApi::NProto::EObjectType::OT_RESOURCE:
            return NLogEvent::OT_RESOURCE;
        case NYP::NClient::NApi::NProto::EObjectType::OT_IP4_ADDRESS_POOL:
            return NLogEvent::OT_IP4_ADDRESS_POOL;
        case NYP::NClient::NApi::NProto::EObjectType::OT_HORIZONTAL_POD_AUTOSCALER:
            return NLogEvent::OT_HORIZONTAL_POD_AUTOSCALER;
        case NYP::NClient::NApi::NProto::EObjectType::OT_DNS_ZONE:
            return NLogEvent::OT_DNS_ZONE;
        default:
            ythrow yexception() << "EObjectType " << NYP::NClient::NApi::NProto::EObjectType_Name(objectType) << " not supported";
    }
}

void TControllerBase::IncrementFactorySensor(const TString& sensor, ui64 x) {
    NON_STATIC_INFRA_RATE_SENSOR_X(GetSensorGroupRef(), sensor, x);
    NON_STATIC_INFRA_RATE_SENSOR_X(GetSensorGroupRef(), GetFactoryName() + "_" + sensor, x);
}

void TControllerBase::InitializeFactorySensors() {
    for (const auto& sensor : FACTORY_RATE_SENSORS_FOR_CONTROLLER_BASE) {
        NON_STATIC_INFRA_RATE_SENSOR_X(GetSensorGroupRef(), sensor, 0);
        NON_STATIC_INFRA_RATE_SENSOR_X(GetSensorGroupRef(), GetFactoryName() + "_" + sensor, 0);
    }
}

TString TControllerBase::GetFactoryName() const {
    return ObjectManagerFactory_->GetFactoryName();
}

size_t TControllerBase::GetCacheSize() const {
    size_t result = 0;
    for (const auto& it : CachedSelectResults_) {
        for (const auto& jt : it.second) {
            result += jt.second.size();
        }
    }
    return result;
}

size_t TControllerBase::GetCachedMatchersSize() const {
    return FilterMatchers_.size();
}

bool TControllerBase::IsResponsibleForLock() const {
    return ObjectManagerFactory_->IsResponsibleForLock();
}

TMaybe<TVector<TClientConfig>> TControllerBase::GetYpClientConfigs() const {
    return ObjectManagerFactory_->GetYpClientConfigs();
}

TDuration TControllerBase::GetSyncInterval() const {
    return SyncInterval_;
}

void TControllerBase::OnGlobalSyncFinish() {
    LeaderSyncDurationSensor_.Reset();
    SyncDurationSensor_.Reset();

    return ObjectManagerFactory_->OnGlobalSyncFinish();
}

size_t TControllerBase::GetShardId() const {
    return ObjectManagerFactory_->GetShard()->GetShardId();
}

size_t TControllerBase::GetNumberOfShards() const {
    return ObjectManagerFactory_->GetShard()->GetNumberOfShards();
}

bool TControllerBase::IsManagedByMaster() const {
    return ObjectManagerFactory_->GetShard()->IsManagedByMaster();
}

bool TControllerBase::RegisterLiveness(
    TLogFramePtr frame,
    const std::function<void()>& onLockAcquired,
    const std::function<void()>& onLockLost
) {
    return ObjectManagerFactory_->GetShard()->GetShardMasterDistributionSubscriber()->RegisterLiveness(frame, onLockAcquired, onLockLost);
}

bool TControllerBase::EnsureTaskPermissionByMaster(TLogFramePtr frame) {
    return ObjectManagerFactory_->GetShard()->GetShardMasterDistributionSubscriber()->EnsureTaskPermissionByMaster(frame);
}

bool TControllerBase::EnsureShardMasterIsAlive(TLogFramePtr frame) {
    return ObjectManagerFactory_->GetShard()->GetShardMasterDistributionSubscriber()->EnsureShardMasterIsAlive(frame);
}

bool TControllerBase::ShouldAbortTaskDueToMasterDistribution(TLogFramePtr frame) {
    return IsManagedByMaster() && !EnsureTaskPermissionByMaster(frame) && EnsureShardMasterIsAlive(frame);
} 

TString TControllerBase::GetFullLeadingInvaderName() const {
    return ObjectManagerFactory_->GetShard()->GetFullLeadingInvaderName();
}

TExpected<void, NLeadingInvader::TError> TControllerBase::EnsureLeading() {
    return ObjectManagerFactory_->GetShard()->EnsureLeading();
}

NLeadingInvader::TLeaderInfo TControllerBase::GetLeaderInfo() const {
    return ObjectManagerFactory_->GetShard()->GetLeaderInfo();
}

void TControllerBase::ResetLeadingInvader(
    const std::function<void()>& onLockAcquired,
    const std::function<void()>& onLockLost,
    bool ignoreIfSet
) {
    ObjectManagerFactory_->GetShard()->ResetLeadingInvader(onLockAcquired, onLockLost, ignoreIfSet);
}

void TControllerBase::DestroyLeadingInvader() {
    ObjectManagerFactory_->GetShard()->DestroyLeadingInvader();
}

const TSensorGroup& TControllerBase::GetSensorGroupRef() {
    return ObjectManagerFactory_->GetSensorGroupRef();
}

TVector<TSelectorResultPtr> AggregateObjects(
    NYP::NClient::TClientPtr client
    , const IObjectManagersFactory::TAggregateArgument& arg
    , TLogFramePtr frame
    , const THashMap<TString, THistogramSensorMap>& histogramSensors
    , const TSensorGroup& sensorGroup
) {
    auto aggregateResults = DoWithRetry<TVector<NYP::NClient::TSelectorResult>, yexception>([&]{
        NON_STATIC_INFRA_RATE_SENSOR(sensorGroup, "yp_aggregate_requests");
        return client->AggregateObjects(
            arg.ObjectType
            , arg.GroupByExpressions
            , arg.AggregateExpressions
            , arg.Filter
            , 0 /* timestamp */
            , MakeHolder<TSensorContext>(
                frame
                , histogramSensors.at(AGGREGATE_OBJECTS)
            )
        ).GetValue(client->Options().Timeout() * 2);
    }, TRetryOptions().WithCount(3).WithSleep(TDuration::Seconds(1)).WithIncrement(TDuration::Seconds(1)), /* throwLast */ true);

    Y_ENSURE(aggregateResults.Defined(), "Failed to aggregate objects. DoWithRetry did not throw an exception and returned an empty TMaybe (impossible situation)");

    TVector<TSelectorResultPtr> result;
    result.reserve(aggregateResults->size());
    for (auto& aggregateResult : *aggregateResults) {
        result.push_back(MakeAtomicShared<NYP::NClient::TSelectorResult>(std::move(aggregateResult)));
    }
    return result;
}

TControllerBase::TWatchObjectsResultsWithInfo TControllerBase::GetUpdatedObjectsIds(
    NYP::NClient::TClientPtr client
    , const IObjectManager::TSelectArgument& selectArgument
    , const ui64 startTimestamp
    , TLogFramePtr frame
) {
    TWatchObjectsResultsWithInfo result;

    NYP::NClient::TWatchObjectsOptions watchOptions;
    if (selectArgument.ClientFilterOptions.Enabled && selectArgument.ClientFilterOptions.WatchSelectorsEnabled) {
        const auto clusterName = GetClusterNameByClient(selectArgument, client);
        const auto& whitelistData = selectArgument.ClientFilterOptions.GetWhitelistData(clusterName);

        NYP::NClient::NApi::NProto::TAttributeSelector watchedPaths;
        watchedPaths.mutable_paths()->Reserve(whitelistData->WatchSelectors.size());
        for (const auto& path : whitelistData->WatchSelectors) {
            watchedPaths.add_paths(path);
        }
        for (const auto& path : selectArgument.Selector) {
            if (!selectArgument.ClientFilterOptions.WatchBannedSelectors.contains(path)) {
                watchedPaths.add_paths(path);
            }
        }
        watchOptions.SetSelector(watchedPaths);
    }

    const size_t eventCountLimit = selectArgument.OverrideYpReqLimitsOptions.Enabled
        ? selectArgument.OverrideYpReqLimitsOptions.ReqWatchLimit
        : selectArgument.Options.Limit();

    TString continuationToken;
    for (;;) {
        TMaybe<NYP::NClient::TWatchObjectsResult> chunk;

        watchOptions.SetStartTimestamp(continuationToken ? 0 : startTimestamp)
                    .SetTimestamp(client->Options().SnapshotTimestamp())
                    .SetEventCountLimit(eventCountLimit)
                    .SetContinuationToken(continuationToken)
                    .SetTimeLimit(WatchObjectsTimeLimit_);
        try {
            chunk = DoWithRetry<NYP::NClient::TWatchObjectsResult, NYP::NClient::TResponseError>(
                [&]{
                    NON_STATIC_INFRA_RATE_SENSOR(GetSensorGroupRef(), "yp_watch_objects_requests");
                    return client->WatchObjects(
                        selectArgument.ObjectType
                        , watchOptions
                        , MakeHolder<TSensorContext>(
                            frame
                            , HistogramSensors_.at(WATCH_OBJECTS)
                        )
                    ).GetValue(client->Options().Timeout() * 2);
                },
                [this, &frame, &selectArgument](const NYP::NClient::TResponseError& ex) {
                    NON_STATIC_INFRA_RATE_SENSOR(GetSensorGroupRef(), "yp_failed_watch_objects_requests");
                    frame->LogEvent(TLOG_WARNING, NLogEvent::TChunkWatchError(
                        NYP::NClient::NApi::NProto::EObjectType_Name(selectArgument.ObjectType)
                        , ex.Address()
                        , ex.RequestId()
                        , ex.Details()
                        , ex.what()
                    ));
                },
                TRetryOptions().WithCount(3).WithSleep(TDuration::Seconds(1)).WithIncrement(TDuration::Seconds(1)),
                /* throwLast */ true
            );
        } catch (NYP::NClient::TResponseError& ex) {
            frame->LogEvent(TLOG_ERR, NLogEvent::TChunkWatchFail(
                NYP::NClient::NApi::NProto::EObjectType_Name(selectArgument.ObjectType)
                , ex.Address()
                , ex.RequestId()
                , ex.Details()
                , ex.what()
            ));
            ythrow ex;
        }
        Y_ENSURE(chunk.Defined(), "Failed to watch objects. DoWithRetry did not throw an exception and returned an empty TMaybe (impossible situation)");

        result.Timestamp = chunk->Timestamp;

        if (chunk->Events.empty()) {
            break;
        }

        continuationToken = chunk->ContinuationToken;

        for (const auto& event : chunk->Events) {
            result.Results.emplace_back(event.object_id());
        }

        if (!eventCountLimit || chunk->Events.size() < eventCountLimit) {
            break;
        }
    }

    return result;
}

TVector<TControllerBase::TObjectSelectResult> TControllerBase::GetObjects(
    NYP::NClient::TClientPtr client
    , const IObjectManager::TSelectArgument& selectArgument
    , const TVector<TString>& objectsIds
    , TLogFramePtr frame
) {
    TVector<TControllerBase::TObjectSelectResult> result;
    result.reserve(objectsIds.size());

    NYP::NClient::TGetObjectOptions getObjectsOptions;
    getObjectsOptions.SetIgnoreNonexistent(true);

    const auto clusterName = GetClusterNameByClient(selectArgument, client);

    TVector<TString> selectors = ExtractAllSelectors(selectArgument, clusterName);

    const size_t eventCountLimit = selectArgument.OverrideYpReqLimitsOptions.Enabled
        ? selectArgument.OverrideYpReqLimitsOptions.ReqGetLimit
        : selectArgument.Options.Limit();

    for (size_t firstBatchIndex = 0; firstBatchIndex < objectsIds.size(); firstBatchIndex += eventCountLimit) {
        const size_t lastBatchIndex = eventCountLimit == 0 ? objectsIds.size() : std::min(objectsIds.size(), firstBatchIndex + eventCountLimit);
        TVector<TString> batchObjectsIds(objectsIds.begin() + firstBatchIndex, objectsIds.begin() + lastBatchIndex);

        try {
            auto objects = DoWithRetry<TVector<NYP::NClient::TSelectorResult>, NYP::NClient::TResponseError>(
                [&, this]{
                    NON_STATIC_INFRA_RATE_SENSOR(GetSensorGroupRef(), "yp_get_objects_requests");
                    return client->GetObjects(
                        selectArgument.ObjectType
                        , batchObjectsIds
                        , selectors
                        , 0 // timestamp (will be skipped in favour of timestamp from options)
                        , getObjectsOptions
                        , MakeHolder<TSensorContext>(
                            frame
                            , HistogramSensors_.at(GET_OBJECTS)
                        )
                    ).GetValue(client->Options().Timeout() * 2);
                },
                [this, &frame, &selectArgument] (const NYP::NClient::TResponseError& ex){
                    NON_STATIC_INFRA_RATE_SENSOR(GetSensorGroupRef(), "yp_failed_get_objects_requests");
                    frame->LogEvent(TLOG_WARNING, NLogEvent::TGetObjectsError(
                        NYP::NClient::NApi::NProto::EObjectType_Name(selectArgument.ObjectType)
                        , ex.Address()
                        , ex.RequestId()
                        , ex.Details()
                        , ex.what()
                    ));
                },
                TRetryOptions().WithCount(3).WithSleep(TDuration::Seconds(1)).WithIncrement(TDuration::Seconds(1)), /* throwLast */ true);

            Y_ENSURE(objects.Defined(), "Failed to get objects. DoWithRetry did not throw an exception and returned an empty TMaybe (impossible situation)");
            for (size_t i = 0; i < batchObjectsIds.size(); ++i) {
                auto& selectorResult = (*objects)[i];
                result.emplace_back(TControllerBase::TObjectSelectResult(batchObjectsIds[i], std::move(selectorResult)));
            }
        } catch (NYP::NClient::TResponseError& ex) {
            frame->LogEvent(TLOG_ERR, NLogEvent::TGetObjectsFail(
                NYP::NClient::NApi::NProto::EObjectType_Name(selectArgument.ObjectType)
                , ex.Address()
                , ex.RequestId()
                , ex.Details()
                , ex.what()
            ));
            ythrow ex;
        }

        if (!eventCountLimit) {
            break;
        }
    }

    return result;
}

TControllerBase::TObjectSelectResultsWithInfo TControllerBase::SelectObjects(
    NYP::NClient::TClientPtr client
    , const IObjectManager::TSelectArgument& selectArgument
    , TLogFramePtr frame
) {
    TVector<TString> selectors = ExtractAllSelectors(selectArgument, GetClusterNameByClient(selectArgument, client));
    const size_t initialSelectorsNumber = selectors.size();

    TMap<TStringBuf, size_t> selectorIdx;
    for (size_t i = 0; i < selectors.size(); ++i) {
        selectorIdx[selectors[i]] = i;
    }

    const TVector<TStringBuf>& keySelectors = NYP::NClient::GetKeySelectors(selectArgument.ObjectType);
    for (const TStringBuf selector : keySelectors) {
        if (selectorIdx.emplace(selector, selectors.size()).second) {
            selectors.emplace_back(selector);
        }
    }

    // Need selctor '/meta/id' to return vector<id, select resut>. Ids is using to update CachedSelectResults_
    static constexpr TStringBuf metaIdSelector = "/meta/id";
    if (selectorIdx.emplace(metaIdSelector, selectors.size()).second) {
        selectors.emplace_back(metaIdSelector);
    }

    TObjectSelectResultsWithInfo result;
    NYP::NClient::TSelectObjectsOptions options = selectArgument.Options;

    size_t totalBytes = 0;
    while (selectArgument.TotalLimit != 0u) {
        if (selectArgument.TotalLimit.Defined() && selectArgument.Options.Limit() != 0u) {
            options.SetLimit(Min(*selectArgument.TotalLimit - result.Results.size(), selectArgument.Options.Limit()));
        }

        TString filter = selectArgument.Filter;
        if (!result.Results.empty()) {
            TVector<TStringBuf> keySelectorValues;
            keySelectorValues.reserve(keySelectors.size());
            for (const TStringBuf keySelector : keySelectors) {
                keySelectorValues.push_back(result.Results.back().SelectResult.Values().value_payloads(selectorIdx[keySelector]).yson());
            }

            TStringBuilder newFilter;
            if (!filter.empty()) {
                newFilter << "(" << filter << ") and ";
            }
            newFilter << NYP::NClient::GetFilterExpression(selectArgument.ObjectType, keySelectorValues);

            filter = std::move(newFilter);
        }

        TMaybe<NYP::NClient::TSelectObjectsResult> chunk;
        try {
            chunk = DoWithRetry<NYP::NClient::TSelectObjectsResult, NYP::NClient::TResponseError>(
                [&, this]{
                    NON_STATIC_INFRA_RATE_SENSOR(GetSensorGroupRef(), "yp_select_requests");
                    return client->SelectObjects(
                        selectArgument.ObjectType
                        , selectors
                        , filter
                        , options
                        , 0 /* timestamp */
                        , MakeHolder<TSensorContext>(
                            frame
                            , HistogramSensors_.at(SELECT_OBJECTS)
                        )
                    ).GetValue(client->Options().Timeout() * 2);
                },
                [this, &frame, &filter, &selectArgument](const NYP::NClient::TResponseError& ex) {
                    NON_STATIC_INFRA_RATE_SENSOR(GetSensorGroupRef(), "yp_failed_select_requests");
                    frame->LogEvent(TLOG_WARNING, NLogEvent::TChunkSelectionError(
                        NYP::NClient::NApi::NProto::EObjectType_Name(selectArgument.ObjectType)
                        , ex.Address()
                        , ex.RequestId()
                        , ex.Details()
                        , filter
                        , ex.what()
                    ));
                },
                TRetryOptions().WithCount(3).WithSleep(TDuration::Seconds(1)).WithIncrement(TDuration::Seconds(1)),
                /* throwLast */ true
            );
        } catch (NYP::NClient::TResponseError& ex) {
            frame->LogEvent(TLOG_ERR, NLogEvent::TChunkSelectionFail(
                NYP::NClient::NApi::NProto::EObjectType_Name(selectArgument.ObjectType)
                , ex.Address()
                , ex.RequestId()
                , ex.Details()
                , filter
                , ex.what()
            ));
            ythrow ex;
        }
        Y_ENSURE(chunk.Defined(), "Failed to select a chunk. DoWithRetry did not throw an exception and returned an empty TMaybe (impossible situation)");

        for (auto& selectResult : chunk->Results) {
            const TString objectId = NYT::NodeFromYsonString(selectResult.Values().value_payloads(selectorIdx[metaIdSelector]).yson()).ConvertTo<TString>();
            result.Results.emplace_back(objectId, std::move(selectResult));

            if (selectArgument.TotalLimit.Defined() && result.Results.size() == *selectArgument.TotalLimit) {
                break;
            }
        }

        result.Timestamp = chunk->Timestamp;

        if (!selectArgument.SelectAll || selectArgument.Options.Limit() == 0 ||
            chunk->Results.size() < selectArgument.Options.Limit() ||
            (selectArgument.TotalLimit.Defined() && result.Results.size() >= *selectArgument.TotalLimit))
        {
            break;
        }
    }

    result.Results.shrink_to_fit();

    if (selectors.size() != initialSelectorsNumber) {
        for (auto& objectSelectResult : result.Results) {
            objectSelectResult.SelectResult.Values().mutable_value_payloads()->Truncate(initialSelectorsNumber);
            if (selectArgument.Options.FetchTimestamps()) {
                objectSelectResult.SelectResult.Values().mutable_timestamps()->Truncate(initialSelectorsNumber);
            }
        }
    }

    for (auto& [ObjectId, SelectResult] : result.Results) {
        for (auto& value_payload : SelectResult.Values().value_payloads()) {
            TStringBuf value = value_payload.yson();
            totalBytes += sizeof(char) * value.size();
        }
    }

    frame->LogEvent(TLOG_DEBUG
        , NLogEvent::TSelectedObjectsSize(
            NYP::NClient::NApi::NProto::EObjectType_Name(selectArgument.ObjectType)
            , totalBytes
        )
    );

    if (selectArgument.TotalLimit.Defined()) {
        Y_ENSURE(result.Results.size() <= *selectArgument.TotalLimit, "Selected " << result.Results.size() << " objects, but total limit is " << *selectArgument.TotalLimit);
    }

    return result;
}

TSelectObjectsResultPtr TControllerBase::LoadObjects(
    NYP::NClient::TClientPtr client
    , const IObjectManager::TSelectArgument& selectArgument
    , TLogFramePtr frame
) {
    // "selectArgument.Options_.FetchTimestamps()", because get objects can not fetch timestamps.
    if ((WatchObjectsBannedObjectsTypes_.contains(selectArgument.ObjectType) || selectArgument.Options.FetchTimestamps() ||
        !selectArgument.SelectAll || !selectArgument.Filter.empty() || selectArgument.TotalLimit.Defined()) && !selectArgument.ClientFilterOptions.Enabled)
    {
        return NonCachedSelectObjects(client, selectArgument, frame);
    }

    if (selectArgument.ClientFilterOptions.Enabled && !selectArgument.ClientFilterOptions.WhitelistEnabled) {
        ythrow yexception() << "Not implemented turning off whitelist check.";
    }

    const TStringBuf clusterName = GetClusterNameByClient(selectArgument, client);
    TVector<TString> valuesOfKeyField = ExtractKeyFieldValue(selectArgument, clusterName);

    try {
        TSelectObjectsResult result = CachedSelectObjects(client, selectArgument, valuesOfKeyField, frame);
        if (selectArgument.ClientFilterOptions.Enabled && !result.Results.empty()) {
            if (auto matcherPtr = FilterMatchers_.FindPtr(selectArgument.ClientFilterOptions.AdditionalFilter); matcherPtr) {
                try {
                    TSimpleTimer timer;

                    ApplyFilterOnSelectResult(result, selectArgument, *matcherPtr);
                    frame->LogEvent(NLogEvent::TClientFilterObjectsTime(
                        selectArgument.ClientFilterOptions.AdditionalFilter,
                        ToString(timer.Get())
                    ));
                    return MakeAtomicShared<TSelectObjectsResult>(result);
                } catch (const NYT::TErrorException& ex) {
                    NON_STATIC_INFRA_RATE_SENSOR(GetSensorGroupRef(), "failed_on_filter_matcher_match");
                    frame->LogEvent(ELogPriority::TLOG_ERR, NLogEvent::TMatchFailed(
                        selectArgument.ClientFilterOptions.AdditionalFilter,
                        ex.what()
                    ));

                    IObjectManager::TSelectArgument fallbackSelectArgument = selectArgument;
                    fallbackSelectArgument.Filter = MergeFilters(selectArgument.Filter, selectArgument.ClientFilterOptions.AdditionalFilter);
                    fallbackSelectArgument.ClientFilterOptions.Enabled = false;
                    return NonCachedSelectObjects(client, fallbackSelectArgument, frame);
                } catch (...) {
                    ythrow yexception() << "Unexpected exception caught on ApplyFilterOnSelectResult.";
                }
            }

            IObjectManager::TSelectArgument fallbackSelectArgument = selectArgument;
            fallbackSelectArgument.Filter = MergeFilters(selectArgument.Filter, selectArgument.ClientFilterOptions.AdditionalFilter);
            fallbackSelectArgument.ClientFilterOptions.Enabled = false;
            return NonCachedSelectObjects(client, fallbackSelectArgument, frame);
        }

        return MakeAtomicShared<TSelectObjectsResult>(result);
    } catch (std::exception& ex) {
        ythrow yexception() << ex.what();
    }
}

void TControllerBase::CachedCreateFilterMatcher(
    NYP::NClient::TClientPtr client
    , const IObjectManager::TSelectArgument& selectArgument
    , TLogFramePtr frame
) {
    const TVector<TString> selectors = ExtractAllSelectors(selectArgument, GetClusterNameByClient(selectArgument, client));
    const TVector<size_t> independentAttributes = GetIndependentAttributesIndices(selectors);
    const TString& additionalFilter = selectArgument.ClientFilterOptions.AdditionalFilter;

    UsedFilterMatchers_.insert(additionalFilter);
    if (auto it = FilterMatchers_.FindPtr(additionalFilter); !it || it->AttributeIndices.size() != independentAttributes.size()) {
        try {
            FilterMatchers_[additionalFilter] = CreateFilterMatcher(selectArgument, additionalFilter, client);
        } catch (const TWhitelistError& e) {
            throw e;
        } catch (const NYT::TErrorException& e) {
            NON_STATIC_INFRA_RATE_SENSOR(GetSensorGroupRef(), "failed_to_create_filter_matcher");
            frame->LogEvent(ELogPriority::TLOG_ERR, NLogEvent::TCreateFilterMatcherFailed(
                additionalFilter
                , e.what()
            ));
        }
    }
}

TSelectObjectsResultPtr TControllerBase::NonCachedSelectObjects(
    NYP::NClient::TClientPtr client
    , const IObjectManager::TSelectArgument& selectArgument
    , TLogFramePtr frame
) {
    auto objectsSelectResults = SelectObjects(client, selectArgument, frame);
    TSelectObjectsResult result;
    for (auto& objectSelectResult : objectsSelectResults.Results) {
        result.Results.emplace_back(MakeAtomicShared<NYP::NClient::TSelectorResult>(std::move(objectSelectResult.SelectResult)));
    }
    result.Timestamp = objectsSelectResults.Timestamp;
    result.Cluster = GetClusterNameByClient(selectArgument, client);
    return MakeAtomicShared<TSelectObjectsResult>(result);
}


void TControllerBase::PrepareCache(
    THashMap<TString, TMaybe<TString>>& objectId2KeyField
    , THashMap<TMaybe<TString>, THashMap<TString, TSelectorResultPtr>>& cachedResults
    , const TVector<TObjectSelectResult>& objects
    , const IObjectManager::TSelectArgument& selectArgument
    , const TStringBuf clusterName
    , bool deleteNonExistantObjects
) {
    const auto& whitelistData = selectArgument.ClientFilterOptions.GetWhitelistData(clusterName);

    for (auto& [objectId, selectorResult] : objects) {
        TMaybe<TString> keyField;
        if (selectArgument.ClientFilterOptions.Enabled && whitelistData->KeyField.Defined()) {
            if (selectorResult.Values().value_payloads().empty()) {
                if (auto objectPtr = objectId2KeyField.FindPtr(objectId); objectPtr) {
                    keyField = objectPtr->GetRef();
                }
            } else {
                auto node = NYT::NodeFromYsonString(selectorResult.Values().value_payloads()[whitelistData->IndexOfKeyField.GetRef() + selectArgument.Selector.size()].yson());
                keyField = node.ConvertTo<TString>();
                objectId2KeyField[objectId] = keyField;
            }
        }

        {
            cachedResults[keyField];
        }

        if (selectorResult.Values().value_payloads().empty() && deleteNonExistantObjects) { // Object does not exist
            cachedResults[keyField].erase(objectId);
            objectId2KeyField.erase(objectId);
        } else {
            cachedResults[keyField][objectId] = MakeAtomicShared<NYP::NClient::TSelectorResult>(std::move(selectorResult));
        }

    }
}


ui64 TControllerBase::CachedSelectObjectsBasedOnSelectObjects(
    NYP::NClient::TClientPtr client
    , const IObjectManager::TSelectArgument& selectArgument
    , THashMap<TString, TMaybe<TString>>& objectId2KeyField
    , THashMap<TMaybe<TString>, THashMap<TString, TSelectorResultPtr>>& cachedResults
    , const TStringBuf clusterName
    , TLogFramePtr frame
) {
    cachedResults.clear();
    objectId2KeyField.clear();

    TSimpleTimer timer;
    auto objectsSelectResults = SelectObjects(client, selectArgument, frame);
    frame->LogEvent(NLogEvent::TSelectObjectsFromCachedModeTime(
        ToString(timer.Get())
    ));
    timer.Reset();
    PrepareCache(objectId2KeyField, cachedResults, objectsSelectResults.Results, selectArgument, clusterName, false /*deleteNonExistantObjects*/);
    frame->LogEvent(NLogEvent::TCacheObjectsTime(
        ToString(timer.Get())
    ));

    return objectsSelectResults.Timestamp;
}


TMaybe<ui64> TControllerBase::CachedSelectObjectsBasedOnWatchObjects(
    NYP::NClient::TClientPtr client
    , const ui64 prevSyncTimestamp
    , const IObjectManager::TSelectArgument& selectArgument
    , THashMap<TString, TMaybe<TString>>& objectId2KeyField
    , THashMap<TMaybe<TString>, THashMap<TString, TSelectorResultPtr>>& cachedResults
    , const TStringBuf clusterName
    , TLogFramePtr frame
) {
    TWatchObjectsResultsWithInfo watchObjectsResults;
    try {
        watchObjectsResults = GetUpdatedObjectsIds(client, selectArgument, prevSyncTimestamp, frame);
        SortUnique(watchObjectsResults.Results);
    } catch (const std::exception& ex) {
        frame->LogEvent(ELogPriority::TLOG_ERR, NLogEvent::TWatchRequestFailed(
            ex.what()
        ));

        return Nothing();
    }

    if (selectArgument.OverrideYpReqLimitsOptions.Enabled &&
        selectArgument.OverrideYpReqLimitsOptions.CriticalGetReqSize &&
        watchObjectsResults.Results.size() > selectArgument.OverrideYpReqLimitsOptions.CriticalGetReqSize
    ) {
        return Nothing();
    }

    auto gotObjects = GetObjects(client, selectArgument, watchObjectsResults.Results, frame);
    PrepareCache(objectId2KeyField, cachedResults, gotObjects, selectArgument, clusterName, true /*deleteNonExistantObjects*/);

    return watchObjectsResults.Timestamp;
}


TSelectObjectsResult TControllerBase::CachedSelectObjects(
    NYP::NClient::TClientPtr client
    , const IObjectManager::TSelectArgument& selectArgument
    , TVector<TString> valuesOfKeyField
    , TLogFramePtr frame
) {
    // If two different threads are executing this code - their client->Options().SnapshotTimestamp()s are equal.

    {
        bool keyPresents = false;
        {
            TReadGuardBase<TLightRWLock> readGuard(CachedSelectResultsLock_);
            keyPresents = CachedSelectResultsLocks_.contains(selectArgument);
        }
        if (!keyPresents) {
            // Guarantee that key is in maps at start of the next block.
            TWriteGuardBase<TLightRWLock> writeGuard(CachedSelectResultsLock_);
            PrevSyncYpTimestamps_[selectArgument];
            CachedSelectResults_[selectArgument];
            ObjectId2KeyField_[selectArgument];
            CachedSelectResultsLocks_[selectArgument];
        }
    }

    frame->LogEvent(NLogEvent::TFactoryCachedSelectWaitingLocks(
        ObjectManagerFactory_->GetFactoryName()
    ));

    TReadGuardBase<TLightRWLock> readGuard(CachedSelectResultsLock_);
    TWriteGuardBase<TLightRWLock> writeGuard(CachedSelectResultsLocks_.at(selectArgument));

    const TStringBuf clusterName = GetClusterNameByClient(selectArgument, client);
    auto& objectId2KeyField = ObjectId2KeyField_.at(selectArgument);
    auto& cachedResults = CachedSelectResults_.at(selectArgument);

    frame->LogEvent(NLogEvent::TFactoryCachedSelectGotLocks(
        ObjectManagerFactory_->GetFactoryName()
    ));

    const auto prevSyncTimestamp = PrevSyncYpTimestamps_.at(selectArgument);
    try {
        if (selectArgument.ClientFilterOptions.Enabled) {
            CachedCreateFilterMatcher(client, selectArgument, frame);
        }

        TSelectObjectsResult result;
        result.Cluster = GetClusterNameByClient(selectArgument, client);

        bool shouldGetObjectsFromYp = !prevSyncTimestamp || *prevSyncTimestamp != client->Options().SnapshotTimestamp();
        bool shouldWatch = prevSyncTimestamp && !ForceAlwaysSelectObjectsForTypes_.contains(selectArgument.ObjectType);

        if (shouldGetObjectsFromYp) {
            if (shouldWatch) {
                if (auto timestamp = CachedSelectObjectsBasedOnWatchObjects(client, *prevSyncTimestamp, selectArgument, objectId2KeyField, cachedResults, clusterName, frame);
                    timestamp.Defined()) {
                    result.Timestamp = timestamp.GetRef();
                } else {
                    result.Timestamp = CachedSelectObjectsBasedOnSelectObjects(client, selectArgument, objectId2KeyField, cachedResults, clusterName, frame);
                }
            } else {
                result.Timestamp = CachedSelectObjectsBasedOnSelectObjects(client, selectArgument, objectId2KeyField, cachedResults, clusterName, frame);
            }
        }

        const auto& whitelistData = selectArgument.ClientFilterOptions.GetWhitelistData(clusterName);
        if (selectArgument.ClientFilterOptions.Enabled && whitelistData->KeyField != Nothing() && !valuesOfKeyField.empty()) {
            for (const auto& valueOfKeyField : valuesOfKeyField) {
                if (auto cachedBatch = CachedSelectResults_.at(selectArgument).FindPtr(valueOfKeyField); cachedBatch) {
                    result.Results.reserve(result.Results.size() + cachedBatch->size());
                    for (const auto& it : *cachedBatch) {
                        result.Results.emplace_back(it.second);
                    }
                }
            }
        } else {
            for (const auto& it : CachedSelectResults_.at(selectArgument)) {
                for (const auto& jt : it.second) {
                    result.Results.emplace_back(jt.second);
                }
            }
        }

        PrevSyncYpTimestamps_.at(selectArgument) = client->Options().SnapshotTimestamp();
        return result;
    } catch (const TWhitelistError& e) {
        throw e;
    } catch (...) {
        CachedSelectResults_.at(selectArgument).clear();
        PrevSyncYpTimestamps_.at(selectArgument) = Nothing();

        throw;
    }
}

TVector<TVector<TString>> TControllerBase::LoadObjects(
    NYP::NClient::TClientPtr client
    , const TVector<NYP::NClient::TObjectAccessAllowedForSubReq>& subReqs
    , TLogFramePtr frame
) {
    if (subReqs.empty()) {
        return {};
    }

    NON_STATIC_INFRA_RATE_SENSOR(GetSensorGroupRef(), "yp_object_access_allowed_for_requests");

    return client->GetObjectAccessAllowedFor(
        subReqs
        , 0
        , MakeHolder<TSensorContext>(
            frame
            , HistogramSensors_.at(GET_OBJECT_ACCESS_ALLOWED_FOR)
        )
    ).GetValue(client->Options().Timeout() * 2);
}

TGetObjectsResultPtr TControllerBase::LoadObjects(
    NYP::NClient::TClientPtr client
    , const IObjectManager::TGetArgument& getArgument
    , TLogFramePtr frame
) {
    NON_STATIC_INFRA_RATE_SENSOR(GetSensorGroupRef(), "yp_get_objects_requests");

    NYP::NClient::TGetObjectOptions options;
    options.SetIgnoreNonexistent(true);

    auto responses = client->GetObjects(
        getArgument.ObjectType
        , getArgument.Ids
        , getArgument.Selector
        , /* timestamp */ 0
        , options
        , MakeHolder<TSensorContext>(
            frame
            , HistogramSensors_.at(GET_OBJECTS)
        )
    ).GetValue(client->Options().Timeout() * 2);

    TGetObjectsResult result;
    result.Results.reserve(responses.size());
    for (auto& response : responses) {
        result.Results.emplace_back(MakeAtomicShared<NYP::NClient::TSelectorResult>(std::move(response)));
    }

    return MakeAtomicShared<TGetObjectsResult>(std::move(result));
}

TVector<TObjectManagerPtr> TControllerBase::GetFactoryObjectManagers(
    const THashMap<TString, NYP::NClient::TClientPtr>& clients
    , TThreadPool& mtpQueue
    , TLogFramePtr frame
) {
    try {
        const TVector<IObjectManagersFactory::TAggregateArgument> aggregateArguments = ObjectManagerFactory_->GetAggregateArgumentsSafe(frame);
        TVector<TVector<TSelectorResultPtr>> aggregateResults(aggregateArguments.size());
        TVector<NThreading::TFuture<void>> aggregateObjectsFutures(aggregateResults.size());
        for (size_t i = 0; i < aggregateResults.size(); ++i) {
            aggregateObjectsFutures[i] = NThreading::Async(
                [this, &clients, frame, &aggregateResults, i, &aggregateArg = std::as_const(aggregateArguments[i])]() {
                    NYP::NClient::TClientPtr client = GetClientByCluster(clients, aggregateArg.ClusterName);
                    aggregateResults[i] = AggregateObjects(
                        client
                        , aggregateArg
                        , frame
                        , HistogramSensors_
                        , GetSensorGroupRef()
                    );

                    frame->LogEvent(NLogEvent::TAggregatedObjects(
                        aggregateResults[i].size()
                        , ConvertYPObjectType(aggregateArg.ObjectType)
                    ));
                }
                , mtpQueue
            );
        }

        NThreading::WaitAll(aggregateObjectsFutures).GetValueSync();

        const TVector<IObjectManager::TSelectArgument> mainObjectSelectArguments = ObjectManagerFactory_->GetSelectArgumentsSafe(aggregateResults, frame);
        TVector<TSelectObjectsResultPtr> selectResults(mainObjectSelectArguments.size());
        TVector<NThreading::TFuture<void>> loadObjectsFutures(selectResults.size());
        for (size_t i = 0; i < selectResults.size(); ++i) {
            loadObjectsFutures[i] = NThreading::Async(
                [this, &clients, frame, &selectResults, i, &selectArg = std::as_const(mainObjectSelectArguments[i])]() {
                    NYP::NClient::TClientPtr client = GetClientByCluster(clients, selectArg.ClusterName);
                    selectResults[i] = LoadObjects(client, selectArg, frame);

                    frame->LogEvent(NLogEvent::TSelectedObjects(
                        selectResults[i]->Results.size()
                        , ConvertYPObjectType(selectArg.ObjectType)
                    ));
                }
                , mtpQueue
            );
        }

        NThreading::WaitAll(loadObjectsFutures).GetValueSync();

        TVector<TExpected<TObjectManagerPtr, IObjectManagersFactory::TValidationError>> results = ObjectManagerFactory_->GetObjectManagersSafe(selectResults, frame);
        TVector<TObjectManagerPtr> objectManagers;
        TMap<std::pair<NYP::NClient::NApi::NProto::EObjectType, TString>, google::protobuf::RepeatedPtrField<TString>> validationErrors; // map<<obj_type, message>, obj_ids>
        for (auto& result : results) {
            if (result) {
                objectManagers.emplace_back(result.Success());
            } else {
                *validationErrors[{result.Error().ObjectType, result.Error().Message}].Add() = result.Error().ObjectId;
            }
        }

        for (auto&& [typeAndMessage, ids] : validationErrors) {
            NLogEvent::TOmittedObjects omittedObjects;
            omittedObjects.SetType(ConvertYPObjectType(typeAndMessage.first));
            omittedObjects.SetMessage(StripString(typeAndMessage.second));
            omittedObjects.MutableIds()->CopyFrom(ids);
            frame->LogEvent(ELogPriority::TLOG_DEBUG, omittedObjects);
        }

        IncrementFactorySensor("selected_objects", results.size());
        IncrementFactorySensor("omitted_object", results.size() - objectManagers.size());

        return objectManagers;

    } catch (yexception& e) {
        NLogEvent::TObjectFactoryLoadError objectFactoryError;
        objectFactoryError.SetFactoryName(GetFactoryName());
        objectFactoryError.SetMessage(e.what());
        frame->LogEvent(ELogPriority::TLOG_ERR, objectFactoryError);

        throw e;
    }
}

IObjectManager::TDependentObjects TControllerBase::LoadAllDependentObjectData(
    const THashMap<TString, NYP::NClient::TClientPtr>& clients
    , TObjectManagerPtr objectManager
    , IThreadPool& auxMtpQueue
    , TLogFramePtr frame
) {
    auto dependentObjectsSelectArguments = objectManager->GetDependentObjectsSelectArguments();
    TVector<TSelectObjectsResultPtr> selectedObjects(dependentObjectsSelectArguments.size());
    TAsyncTaskBatch taskBatch(&auxMtpQueue);

    for (size_t i = 0; i < selectedObjects.size(); ++i) {
        taskBatch.Add(
            [this, &clients, i, &selectedObjects, frame, &selectArgument = std::as_const(dependentObjectsSelectArguments[i])]() {
                NYP::NClient::TClientPtr client = GetClientByCluster(clients, selectArgument.ClusterName);
                selectedObjects[i] = LoadObjects(client, selectArgument, frame);
            }
        );
    }

    auto accessAllowedObjectsArguments = objectManager->GetObjectAccessAllowedForArgumentsWithClusters();
    TVector<TVector<TVector<TString>>> accessAllowedObjects(accessAllowedObjectsArguments.size());

    for (size_t i = 0; i < accessAllowedObjects.size(); ++i) {
        const auto& subReqs = accessAllowedObjectsArguments[i].first;
        const auto& clusterName = accessAllowedObjectsArguments[i].second;
        taskBatch.Add(
            [this, &clients, i, &accessAllowedObjects, &subReqs, &clusterName, frame]() {
                NYP::NClient::TClientPtr client = GetClientByCluster(clients, clusterName);
                accessAllowedObjects[i] = LoadObjects(client, subReqs, frame);
            }
        );
    }

    auto dependentObjectsGetArguments = objectManager->GetDependentObjectsGetArguments();
    TVector<TGetObjectsResultPtr> gotObjects(dependentObjectsGetArguments.size());

    for (size_t i = 0; i < gotObjects.size(); ++i) {
        taskBatch.Add(
            [this, &clients, i, &gotObjects, frame, &getArgument = std::as_const(dependentObjectsGetArguments[i])]() {
                NYP::NClient::TClientPtr client = GetClientByCluster(clients, Nothing());
                gotObjects[i] = LoadObjects(client, getArgument, frame);
            }
        );
    }

    auto waitInfo = taskBatch.WaitAllAndProcessNotStarted();
    if (waitInfo.ExceptionPtr) {
        std::rethrow_exception(waitInfo.ExceptionPtr);
    }
    return {std::move(selectedObjects), std::move(accessAllowedObjects), std::move(gotObjects)};
}

void TControllerBase::SyncObjectOnCluster(
    NYP::NClient::TTransactionFactoryPtr transactionFactory
    , const TString& clusterName
    , const TString& objectId
    , const TVector<IObjectManager::TRequest>& requests
    , const TDuration& timeout
    , const ui32 ypRequestsBatchSize
    , TLogFramePtr frame
) {
        if (!IsResponsibleForLock()) {
            if (auto leadingResult = EnsureLeading(); !(bool)leadingResult) {
                ythrow yexception() << "Lost lock '" << leadingResult.Error().Reason << "'";
            }
        }

        {
            size_t createCnt = 0;
            size_t removeCnt = 0;
            size_t updateCnt = 0;
            for (const auto& req : requests) {
                createCnt += std::holds_alternative<NYP::NClient::TCreateObjectRequest>(req);
                removeCnt += std::holds_alternative<NYP::NClient::TRemoveObjectRequest>(req);
                updateCnt += std::holds_alternative<NYP::NClient::TUpdateRequest>(req);
            }
            frame->LogEvent(TLOG_DEBUG, NLogEvent::TObjectUpdate(objectId, createCnt, removeCnt, updateCnt, clusterName));
        }

        TSimpleTimer timer;
        auto transaction = transactionFactory->CreateTransaction();
        const TString transactionId = transaction->TransactionId();
        frame->LogEvent(TLOG_DEBUG, NLogEvent::TCreateTransaction(transactionFactory->MasterAddress(), transactionId, clusterName));

        try {
            TVector<NYP::NClient::TCreateObjectRequest> create;
            TVector<NYP::NClient::TRemoveObjectRequest> remove;
            TVector<NYP::NClient::TUpdateRequest> update;
            for (auto it = requests.begin(); it != requests.end(); ++it) {
                auto next = it + 1;

                if (std::holds_alternative<NYP::NClient::TCreateObjectRequest>(*it)) {
                    create.emplace_back(std::move(std::get<NYP::NClient::TCreateObjectRequest>(*it)));
                    if (create.size() >= ypRequestsBatchSize || next == requests.end() ||
                        !std::holds_alternative<NYP::NClient::TCreateObjectRequest>(*next)) {
                        SetYpReqCountSensors(create, "create");

                        const TVector<TString> rsp = transaction->CreateObjects(create).GetValue(timeout);
                        if (frame->AcceptLevel(TLOG_DEBUG)) {
                            Y_ENSURE(rsp.size() == create.size(), "response size not equal to request size");
                            for (size_t i = 0; i < rsp.size(); ++i) {
                                frame->LogEvent(TLOG_DEBUG, NLogEvent::TCreateYPObject(
                                    rsp[i]
                                    , ConvertYPObjectType(create[i].GetObjectType())
                                    , NJson2Yson::ConvertYson2Json(create[i].GetAttributes())
                                    , clusterName
                                    , GetFactoryName()
                                ));
                            }
                        }

                        create.clear();
                    }
                } else if (std::holds_alternative<NYP::NClient::TRemoveObjectRequest>(*it)) {
                    remove.emplace_back(std::move(std::get<NYP::NClient::TRemoveObjectRequest>(*it)));
                    if (remove.size() >= ypRequestsBatchSize || next == requests.end() ||
                        !std::holds_alternative<NYP::NClient::TRemoveObjectRequest>(*next)) {
                        SetYpReqCountSensors(remove, "remove");

                        try {
                            transaction->RemoveObjects(remove).GetValue(timeout);
                            if (frame->AcceptLevel(TLOG_DEBUG)) {
                                for (auto& it : remove) {
                                    frame->LogEvent(TLOG_DEBUG, NLogEvent::TRemoveYPObject(
                                        it.GetObjectId()
                                        , ConvertYPObjectType(it.GetObjectType())
                                        , clusterName
                                        , GetFactoryName()
                                    ));
                                }
                            }
                        } catch (const NYP::NClient::TResponseError& e) {
                            const int errorCode = static_cast<int>(e.Error().GetNonTrivialCode());
                            switch (NYP::NClient::NApi::EErrorCode(errorCode)) {
                                case NYP::NClient::NApi::EErrorCode::NoSuchObject:
                                    IncrementFactorySensor("remove_objects_failed.no_such_object", 1);
                                    frame->LogEvent(TLOG_ERR, NLogEvent::TRemoveObjectsFailed(CurrentExceptionMessage()));
                                    break;
                                default:
                                    throw e;
                            }
                        }

                        remove.clear();
                    }
                } else if (std::holds_alternative<NYP::NClient::TUpdateRequest>(*it)) {
                    update.emplace_back(std::move(std::get<NYP::NClient::TUpdateRequest>(*it)));
                    if (update.size() >= ypRequestsBatchSize || next == requests.end() ||
                        !std::holds_alternative<NYP::NClient::TUpdateRequest>(*next)) {
                        SetYpReqCountSensors(update, "update");

                        transaction->UpdateObjects(update).GetValue(timeout);
                        if (frame->AcceptLevel(TLOG_DEBUG)) {
                            for (auto& it : update) {
                                auto msg = NLogEvent::TUpdateYPObject(
                                    it.GetObjectId()
                                    , ConvertYPObjectType(it.GetObjectType())
                                    , {}
                                    , {}
                                    , clusterName
                                    , GetFactoryName()
                                );
                                for (auto& setRequest: it.GetSetVec()) {
                                    NLogEvent::TUpdateYPObject_TSetRequest* logged = msg.AddSetRequests();
                                    logged->SetPath(setRequest.GetPath());
                                    logged->SetValue(NJson2Yson::ConvertYson2Json(setRequest.GetValue()));
                                    logged->SetRecursive(setRequest.GetRecursive());
                                }
                                for (auto& removeRequest: it.GetRemoveVec()) {
                                    NLogEvent::TUpdateYPObject_TRemoveRequest* logged = msg.AddRemoveRequests();
                                    logged->SetPath(removeRequest.GetPath());
                                }
                                frame->LogEvent(TLOG_DEBUG, msg);
                            }
                        }

                        update.clear();
                    }
                } else {
                    ythrow yexception() << "Unexpected type index: '" << it->index()  << "'";
                }
            }

            transaction->Commit().GetValue(timeout);

            NON_STATIC_INFRA_RATE_SENSOR(GetSensorGroupRef(), "successful_transactions");
            frame->LogEvent(TLOG_DEBUG, NLogEvent::TTransactionCommitSuccess(transactionId, clusterName));
        } catch (yexception& e) {
            NON_STATIC_INFRA_RATE_SENSOR(GetSensorGroupRef(), "failed_transactions");
            frame->LogEvent(TLOG_ERR, NLogEvent::TTransactionCommitError(transactionId, CurrentExceptionMessage(), clusterName));
            throw e;
        }

        const auto spent = timer.Get();
        frame->LogEvent(NLogEvent::TSyncedObject(objectId, ToString(spent), clusterName));
}

void TControllerBase::SyncObject(
    TObjectManagerPtr objectManager
    , const IObjectManager::TDependentObjects& dependentObjects
    , const THashMap<TString, NYP::NClient::TTransactionFactoryPtr>& transactionFactories
    , const THashMap<TString, NYP::NClient::TClientPtr>& clients
    , const ui32 ypRequestsBatchSize
    , IThreadPool& auxMtpQueue
    , TLogFramePtr frame
    , ui32 retryCount
) {
    THashMap<TString, TVector<IObjectManager::TRequest>> requestsByCluster;
    for (const auto& [clusterName, _] : clients) {
        requestsByCluster[clusterName] = TVector<IObjectManager::TRequest>();
    }
    objectManager->GenerateYpUpdates(
        dependentObjects
        , requestsByCluster
        , frame
    );

    const TString objectId = objectManager->GetObjectId();

    bool hasUpdates = false;
    for (auto& requestsOnSingleCluster : requestsByCluster) {
        hasUpdates |= !requestsOnSingleCluster.second.empty();
    }

    if (!hasUpdates) {
        frame->LogEvent(TLOG_DEBUG, NLogEvent::TEmptyObjectUpdate(objectId));
        return; // nothing to update
    }

    for (const auto& [clusterName, _] : requestsByCluster) {
        if (!transactionFactories.contains(clusterName)) {
            throw yexception() << "Unknown cluster name " << clusterName;
        }
    }

    TAsyncTaskBatch taskBatch(&auxMtpQueue);

    for (const auto& [clusterName, requests] : requestsByCluster) {
        taskBatch.Add(
            [this, &transactionFactories, &clients, objectManager, ypRequestsBatchSize, frame, &clusterName = clusterName, &requests = requests, &objectId, retryCount]() {
                auto transactionFactory = transactionFactories.at(clusterName);
                auto client = clients.at(clusterName);

                auto sync = [this, transactionFactory, client, ypRequestsBatchSize, frame, &clusterName = clusterName, &requests = requests, &objectId]() {
                    SyncObjectOnCluster(
                        transactionFactory,
                        clusterName,
                        objectId,
                        requests,
                        client->Options().Timeout() * 2,
                        ypRequestsBatchSize,
                        frame
                    );
                };

                std::function<void(const yexception&)> onFail = [this, frame, client, objectManager, &clusterName = clusterName](const yexception& e) {
                    NLogEvent::TSyncTryFailed event;
                    event.SetObjectId(objectManager->GetObjectId());
                    event.SetMessage(e.what());
                    event.SetClusterName(clusterName);
                    frame->LogEvent(ELogPriority::TLOG_ERR, event);

                    IncrementFactorySensor("sync_object_on_cluster_retries", 1);

                    /* If YP masters are dead - object managers will fail requests
                    * with full timeout, this takes long time.
                    * So rediscovering the masters helps to avoid this.
                    */
                    client->ReconstructBalancing(
                        {}
                        , MakeHolder<TSensorContext>(
                            frame
                            , HistogramSensors_.at(RECONSTRUCT_BALANCING)
                        )
                    );
                };
                try {
                    DoWithRetry(sync, onFail, TRetryOptions().WithCount(retryCount), /* throwLast */ true);
                    IncrementFactorySensor("synced_objects_on_cluster", 1);
                } catch (const yexception& e) {
                    NLogEvent::TSyncTryFailed event;
                    event.SetObjectId(objectManager->GetObjectId());
                    event.SetMessage(TString(e.what()) + ", maximum retries achieved");
                    event.SetClusterName(clusterName);
                    frame->LogEvent(ELogPriority::TLOG_ERR, event);
                }
            }
        );
    }
    auto waitInfo = taskBatch.WaitAllAndProcessNotStarted();
    if (waitInfo.ExceptionPtr) {
        std::rethrow_exception(waitInfo.ExceptionPtr);
    }
}

NYP::NClient::TClientPtr TControllerBase::GetClientByCluster(const THashMap<TString, NYP::NClient::TClientPtr>& clients, const TMaybe<TString>& clusterName) const {
    if (!clusterName) {
        if (clients.size() == 1) {
            return clients.begin()->second;
        } else {
            ythrow yexception() << "Cluster name is undefined and there're several clusters in config";
        }
    } else {
        if (auto clientPtr = clients.FindPtr(*clusterName)) {
            return *clientPtr;
        } else {
            ythrow yexception() << "In config there's no cluster " + *clusterName;
        }
    }
}

} // namespace NInfra::NController

