#pragma once

#include "client_filter_helpers.h"
#include "sensor_info.h"

#include <infra/libs/leading_invader/leading_invader.h>

#include <util/system/rwlock.h>

namespace NInfra::NController {

namespace {

constexpr std::initializer_list<TStringBuf> HISTOGRAMS_SENSOR_TYPES_FOR_CONTROLLER_BASE = {
    DURATION,
};

constexpr std::initializer_list<TStringBuf> HISTOGRAMS_REQUEST_TYPES_FOR_CONTROLLER_BASE = {
    RECONSTRUCT_BALANCING,
    SELECT_OBJECTS,
    WATCH_OBJECTS,
    GET_OBJECTS,
    GET_OBJECT_ACCESS_ALLOWED_FOR,
    AGGREGATE_OBJECTS,
};

constexpr std::initializer_list<TStringBuf> FACTORY_RATE_SENSORS_FOR_CONTROLLER_BASE = {
    FAILED_SYNC_CYCLES,
    OMITTED_OBJECTS,
    REMOVE_OBJECTS_FAILED_NO_SUCH_OBJECT,
    SELECTED_OBJECTS,
    SUCCESSFUL_SYNC_CYCLES,
    SYNC_CYCLES,
    SYNC_OBJECT_ERRORS,
    SYNC_OBJECT_RETRIES,
    SYNC_OBJECTS_ON_CLUSTER_RETRIES,
    SYNCED_OBJECTS_ON_CLUSTER,
};

} // namespace

class TControllerBase;
using TControllerPtr = TAtomicSharedPtr<TControllerBase>;

class TControllerBase {
public:
    struct TObjectSelectResult {
        TObjectSelectResult(
            const TString& objectId
            , NYP::NClient::TSelectorResult&& selectResult
        )
            : ObjectId(objectId)
            , SelectResult(selectResult)
        {}

        TString ObjectId;
        NYP::NClient::TSelectorResult SelectResult;
    };

    struct TObjectSelectResultsWithInfo {
        TVector<TObjectSelectResult> Results;
        ui64 Timestamp = 0u;
    };

    struct TWatchObjectsResultsWithInfo {
        TVector<TString> Results;
        ui64 Timestamp = 0u;
    };

public:
    TControllerBase(
        TObjectManagersFactoryPtr objectManagerFactory
        , const TDuration defaultSyncInterval
        , const TDuration watchObjectsTimeLimit
        , const TSet<TString>& watchObjectsBannedObjectsTypes
        , const TSet<TString>& forceAlwaysSelectObjectsForTypes
    )
        : ObjectManagerFactory_(objectManagerFactory)
        , SyncInterval_(objectManagerFactory->GetCustomSyncInterval().GetOrElse(defaultSyncInterval))
        , WatchObjectsTimeLimit_(watchObjectsTimeLimit)
        , LeaderSyncDurationSensor_(ObjectManagerFactory_->GetSensorGroupRef(), GetFactoryName() + "_leader_sync_loop_time")
        , SyncDurationSensor_(ObjectManagerFactory_->GetSensorGroupRef(), GetFactoryName() + "_sync_loop_time")
        , HistogramSensors_(
            GenerateHistogramSensors(
                HISTOGRAMS_REQUEST_TYPES_FOR_CONTROLLER_BASE
                , HISTOGRAMS_SENSOR_TYPES_FOR_CONTROLLER_BASE
                , TVector<std::pair<TStringBuf, TStringBuf>>({
                    {ORIGIN, CONTROLLER_BASE}
                    , {FACTORY_NAME, objectManagerFactory->GetFactoryName()}
                })
            )
        )
    {
        InitializeFactorySensors();

        for (const auto& objTypeStr : watchObjectsBannedObjectsTypes) {
            static const auto* type = NYT::NYson::ReflectProtobufEnumType(NYP::NClient::NApi::NProto::EObjectType_descriptor());
            const auto objTypeOpt = NYT::NYson::FindProtobufEnumValueByLiteral<NYP::NClient::NApi::NProto::EObjectType>(type, objTypeStr);
            Y_ENSURE(objTypeOpt, TStringBuf() << "Unknown object type '" << objTypeStr << "'");
            WatchObjectsBannedObjectsTypes_.emplace(*objTypeOpt);
        }

        for (const auto& objTypeStr : forceAlwaysSelectObjectsForTypes) {
            static const auto* type = NYT::NYson::ReflectProtobufEnumType(NYP::NClient::NApi::NProto::EObjectType_descriptor());
            const auto objTypeOpt = NYT::NYson::FindProtobufEnumValueByLiteral<NYP::NClient::NApi::NProto::EObjectType>(type, objTypeStr);
            Y_ENSURE(objTypeOpt, TStringBuf() << "Unknown object type '" << objTypeStr << "'");
            ForceAlwaysSelectObjectsForTypes_.emplace(*objTypeOpt);
        }
    }

    void Sync(
        const THashMap<TString, NYP::NClient::TClientPtr>& clients
        , const THashMap<TString, NYP::NClient::TTransactionFactoryPtr>& transactionFactories
        , TThreadPool& mtpQueue
        , IThreadPool& auxMtpQueue
        , TLogFramePtr frame
        , ui32 retryCount
        , const ui32 ypRequestsBatchSize
        , const ui32 maxSeqFailedObjMngrsCount
    );

    void IncrementFactorySensor(
        const TString& sensor
        , ui64 x
    );

    TString GetFactoryName() const;

    size_t GetCacheSize() const;

    size_t GetCachedMatchersSize() const;

    bool IsResponsibleForLock() const;

    TMaybe<TVector<TClientConfig>> GetYpClientConfigs() const;

    TDuration GetSyncInterval() const;

    void OnGlobalSyncFinish();

    // sets controller's leadereship state
    void SetLeadership(const bool isLeader /* acquiring lock or not */) {
        ObjectManagerFactory_->SetLeadership(isLeader);
    }

    size_t GetShardId() const;

    size_t GetNumberOfShards() const;

    bool IsManagedByMaster() const;

    bool RegisterLiveness(
        TLogFramePtr frame,
        const std::function<void()>& onLockAcquired = []{},
        const std::function<void()>& onLockLost = []{}
    );

    bool EnsureTaskPermissionByMaster(TLogFramePtr frame);

    bool EnsureShardMasterIsAlive(TLogFramePtr frame);

    bool ShouldAbortTaskDueToMasterDistribution(TLogFramePtr frame);

    TString GetFullLeadingInvaderName() const;

    TExpected<void, NLeadingInvader::TError> EnsureLeading();

    NLeadingInvader::TLeaderInfo GetLeaderInfo() const;

    void ResetLeadingInvader(
        const std::function<void()>& onLockAcquired = []{},
        const std::function<void()>& onLockLost = []{},
        bool ignoreIfSet = false
    );

    void DestroyLeadingInvader();

    const TSensorGroup& GetSensorGroupRef();

private:
    NLogEvent::EYPObjectType ConvertYPObjectType(
        NYP::NClient::NApi::NProto::EObjectType objectType
    );

    void SyncObjectOnCluster(
        NYP::NClient::TTransactionFactoryPtr transactionFactory
        , const TString& clusterName
        , const TString& objectId
        , const TVector<IObjectManager::TRequest>& requests
        , const TDuration& timeout
        , const ui32 ypRequestsBatchSize
        , TLogFramePtr frame
    );

    void SyncObject(
        TObjectManagerPtr objectManager
        , const IObjectManager::TDependentObjects& dependentObjects
        , const THashMap<TString, NYP::NClient::TTransactionFactoryPtr>& transactionFactories
        , const THashMap<TString, NYP::NClient::TClientPtr>& clients
        , const ui32 ypRequestsBatchSize
        , IThreadPool& auxMtpQueue
        , TLogFramePtr frame
        , ui32 retryCount
    );

    TObjectSelectResultsWithInfo SelectObjects(
        NYP::NClient::TClientPtr client
        , const IObjectManager::TSelectArgument& selectArgument
        , TLogFramePtr frame
    );

    TWatchObjectsResultsWithInfo GetUpdatedObjectsIds(
        NYP::NClient::TClientPtr client
        , const IObjectManager::TSelectArgument& selectArgument
        , const ui64 startTimestamp
        , TLogFramePtr frame
    );

    TVector<TObjectSelectResult> GetObjects(
        NYP::NClient::TClientPtr client
        , const IObjectManager::TSelectArgument& selectArgument
        , const TVector<TString>& objectIds
        , TLogFramePtr frame
    );

    void CachedCreateFilterMatcher(
        NYP::NClient::TClientPtr client
        , const IObjectManager::TSelectArgument& selectArgument
        , TLogFramePtr frame
    );

    TSelectObjectsResultPtr NonCachedSelectObjects(
        NYP::NClient::TClientPtr client
        , const IObjectManager::TSelectArgument& selectArgument
        , TLogFramePtr frame
    );

    void PrepareCache(
        THashMap<TString, TMaybe<TString>>& objectId2KeyField
        , THashMap<TMaybe<TString>, THashMap<TString, TSelectorResultPtr>>& cachedResults
        , const TVector<TObjectSelectResult>& objects
        , const IObjectManager::TSelectArgument& selectArgument
        , const TStringBuf clusterName
        , bool deleteNonExistantObjects
    );

    ui64 CachedSelectObjectsBasedOnSelectObjects(
        NYP::NClient::TClientPtr client
        , const IObjectManager::TSelectArgument& selectArgument
        , THashMap<TString, TMaybe<TString>>& objectId2KeyField
        , THashMap<TMaybe<TString>, THashMap<TString, TSelectorResultPtr>>& cachedResults
        , const TStringBuf clusterName
        , TLogFramePtr frame
    );

    TMaybe<ui64> CachedSelectObjectsBasedOnWatchObjects(
        NYP::NClient::TClientPtr client
        , const ui64 prevSyncTimestamp
        , const IObjectManager::TSelectArgument& selectArgument
        , THashMap<TString, TMaybe<TString>>& objectId2KeyField
        , THashMap<TMaybe<TString>, THashMap<TString, TSelectorResultPtr>>& cachedResults
        , const TStringBuf clusterName
        , TLogFramePtr frame
    );

    TSelectObjectsResult CachedSelectObjects(
        NYP::NClient::TClientPtr client
        , const IObjectManager::TSelectArgument& selectArgument
        , TVector<TString> valuesOfKeyField
        , TLogFramePtr frame
    );

    TSelectObjectsResultPtr LoadObjects(
        NYP::NClient::TClientPtr client
        , const IObjectManager::TSelectArgument& selectArgument
        , TLogFramePtr frame
    );

    TVector<TVector<TString>> LoadObjects(
        NYP::NClient::TClientPtr client
        , const TVector<NYP::NClient::TObjectAccessAllowedForSubReq>& subReqs
        , TLogFramePtr frame
    );

    TGetObjectsResultPtr LoadObjects(
        NYP::NClient::TClientPtr client
        , const IObjectManager::TGetArgument& getArgument
        , TLogFramePtr frame
    );

    /*
        @return TObjectManagerPtr of all valid objects
        invalid objects are omitted
    */
    TVector<TObjectManagerPtr> GetFactoryObjectManagers(
        const THashMap<TString, NYP::NClient::TClientPtr>& clients
        , TThreadPool& mtpQueue
        , TLogFramePtr frame
    );

    IObjectManager::TDependentObjects LoadAllDependentObjectData(
        const THashMap<TString, NYP::NClient::TClientPtr>& clients
        , TObjectManagerPtr objectManager
        , IThreadPool& auxMtpQueue
        , TLogFramePtr frame
    );

    template<typename T>
    void SetYpReqCountSensors(const TVector<T>& requests, const TString& reqType) {
        TMap<NYP::NClient::NApi::NProto::EObjectType, ui32> objTypeCount;
        for (const auto& req : requests) {
            ++objTypeCount[req.GetObjectType()];
        }

        static const auto* objTypeDesciptor = NYT::NYson::ReflectProtobufEnumType(NYP::NClient::NApi::NProto::EObjectType_descriptor());
        for (const auto& [objType, cnt] : objTypeCount) {
            const TString objTypeStr = TString{NYT::NYson::FindProtobufEnumLiteralByValue(objTypeDesciptor, objType)};
            IncrementFactorySensor(objTypeStr + "." + reqType, cnt);
        }
    }

    NYP::NClient::TClientPtr GetClientByCluster(const THashMap<TString, NYP::NClient::TClientPtr>& clients, const TMaybe<TString>& clusterName) const;

    void InitializeFactorySensors();

private:
    TObjectManagersFactoryPtr ObjectManagerFactory_;
    TAtomic NeedAbort_ = 0;
    const TDuration SyncInterval_;
    const TDuration WatchObjectsTimeLimit_;
    TSet<NYP::NClient::NApi::NProto::EObjectType> WatchObjectsBannedObjectsTypes_;
    TSet<NYP::NClient::NApi::NProto::EObjectType> ForceAlwaysSelectObjectsForTypes_;
    TDurationSensor LeaderSyncDurationSensor_;
    TDurationSensor SyncDurationSensor_;
    const THashMap<TString, THistogramSensorMap> HistogramSensors_;

    // <select argument, <TMaybe<key field>, <object id, select results>>>
    TMap<IObjectManager::TSelectArgument, THashMap<TMaybe<TString>, THashMap<TString, TSelectorResultPtr>>> CachedSelectResults_;
    TMap<IObjectManager::TSelectArgument, THashMap<TString, TMaybe<TString>>> ObjectId2KeyField_;
    THashMap<TString, TCachedFilterMatcher> FilterMatchers_;
    THashSet<TString> UsedFilterMatchers_;
    TMap<IObjectManager::TSelectArgument, TMaybe<ui64>> PrevSyncYpTimestamps_;
    TMap<IObjectManager::TSelectArgument, TLightRWLock> CachedSelectResultsLocks_;
    TLightRWLock CachedSelectResultsLock_;
};

} // namespace NInfra::NController
