#pragma once

#include "controller.h"

#include <infra/libs/controller/config/config.pb.h>

#include <infra/libs/leading_invader/leading_invader.h>
#include <infra/libs/logger/logger.h>

#include <infra/libs/updatable_proto_config/accessor.h>

#include <util/system/thread.h>

namespace NInfra::NController {

namespace {

constexpr std::initializer_list<TStringBuf> HISTOGRAMS_SENSOR_TYPES_FOR_CONTROLLER_LOOP = {
    DURATION,
};

constexpr std::initializer_list<TStringBuf> HISTOGRAMS_REQUEST_TYPES_FOR_CONTROLLER_LOOP = {
    RECONSTRUCT_BALANCING,
    GENERATE_TIMESTAMP,
};

} // namespace

using TGetActualConfig = std::function<std::pair<const TControllerConfig&, bool>()>;

class TControllerLoop {
public:
    TControllerLoop(NUpdatableProtoConfig::TAccessor<TControllerConfig> config, TLogger& logger, TVector<TControllerPtr> controllers);

    void RunSyncLoop();

    void Shutdown();

    void ReopenLogs();

    bool LockAcquired(const TMaybe<TString>& shardLeadingInvaderName = Nothing()) const;

    NLeadingInvader::TLeaderInfo GetLeaderInfo(TMaybe<TString> shardLeadingInvaderName = Nothing()) const;

private:
    void SyncController(
        TControllerPtr controller
        , const TControllerConfig& actualConfig
        , const THashMap<TString, NYP::NClient::TClientPtr>& clients
        , const THashMap<TString, NYP::NClient::TTransactionFactoryPtr>& transactionFactories
    );

    void ControllerSyncLoop(
        TControllerPtr controller
        , TGetActualConfig getActualConfig
    );

    static void* RunControllerSyncLoop(
        void* data
    );

private:
    class TSequentialFailsCounter {
    public:
        TSequentialFailsCounter(const ui32 maxFailsCount);

        void FeedbackSuccess(const TString& factoryName);

        void FeedbackFailure(const TString& factoryName);

        bool IsThresholdExceeded(const TString& factoryName, const size_t numberOfShards) const;

        void DoubleThresholdAndReset(const TString& factoryName, const size_t numberOfShards);

    private:
        ui32 MaxFailsCount_;
        TMap<TString, ui32> SequentialFailsCount_; // <factory name, count>
        TLightRWLock Lock_;
    };

private:
    NUpdatableProtoConfig::TAccessor<TControllerConfig> Config_;
    TAtomic Running_;
    TLogger& Logger_;
    TVector<TControllerPtr> Controllers_;
    THashMap<TString, TThreadPool> MtpQueues_;
    THolder<IThreadPool> AuxMtpQueue_;
    THashMap<TString, TAtomic> LockAcquired_;
    THashMap<TString, TAtomic> LivenessLockAcquired_;
    TSequentialFailsCounter SequentialFailsCounter_;
    const THashMap<TString, THistogramSensorMap> HistogramSensors_;
};

} // namespace NInfra::NController
