#pragma once

#include <infra/libs/controller/object_manager/object_manager.h>
#include <infra/libs/logger/log_frame.h>
#include <infra/libs/sensors/sensor.h>
#include <yp/cpp/yp/request_model.h>

#include <util/datetime/cputimer.h>

namespace NInfra::NController {

using THistogramSensorMap = THashMap<TString, TAtomicSharedPtr<NInfra::THistogramRateSensor>>;

namespace {

constexpr TStringBuf ORIGIN = "origin";
constexpr TStringBuf FACTORY_NAME = "factory_name";
constexpr TStringBuf REQUEST_TYPE = "request_type";

constexpr TStringBuf DURATION = "duration_mcs";
constexpr TStringBuf CONTROLLER_LOOP = "TControllerLoop";
constexpr TStringBuf CONTROLLER_BASE = "TControllerBase";

constexpr TStringBuf RECONSTRUCT_BALANCING = "reconstruct_balancing";
constexpr TStringBuf GENERATE_TIMESTAMP = "generate_timestamp";
constexpr TStringBuf SELECT_OBJECTS = "select_objects";
constexpr TStringBuf WATCH_OBJECTS = "watch_objects";
constexpr TStringBuf GET_OBJECTS = "get_objects";
constexpr TStringBuf GET_OBJECT_ACCESS_ALLOWED_FOR = "get_object_access_allowed_for";
constexpr TStringBuf AGGREGATE_OBJECTS = "aggregate_objects";

constexpr TStringBuf FAILED_SYNC_CYCLES = "failed_sync_cycles";
constexpr TStringBuf OMITTED_OBJECTS = "omitted_object";
constexpr TStringBuf REMOVE_OBJECTS_FAILED_NO_SUCH_OBJECT = "remove_objects_failed.no_such_object";
constexpr TStringBuf SELECTED_OBJECTS = "selected_objects";
constexpr TStringBuf SUCCESSFUL_SYNC_CYCLES = "successful_sync_cycles";
constexpr TStringBuf SYNC_CYCLES = "sync_cycles";
constexpr TStringBuf SYNC_OBJECT_ERRORS = "sync_object_errors";
constexpr TStringBuf SYNC_OBJECT_RETRIES = "sync_object_retries";
constexpr TStringBuf SYNC_OBJECTS_ON_CLUSTER_RETRIES = "sync_object_on_cluster_retries";
constexpr TStringBuf SYNCED_OBJECTS_ON_CLUSTER = "synced_objects_on_cluster";

constexpr double BASE_OF_HISTOGRAM = 1.5;
constexpr double SCALE_OF_HISTOGRAM = 10.0;

class TSensorContext: public NYP::NClient::TRequestContext {
public:
    TSensorContext(TLogFramePtr frame, THistogramSensorMap sensors)
        : Frame_(frame)
        , Sensors_(sensors)
    {
    }

    void BeforeRequestCallback(const NYP::NClient::TRequestInfo& requestInfo) override {
        const TString yson = NYP::NClient::ObjectToYson(*requestInfo.RequestPtr, NYT::NYson::EYsonFormat::Text);
        Frame_->LogEvent(ELogPriority::TLOG_DEBUG, NLogEvent::TYpClientRequestStart(requestInfo.Id, requestInfo.RequestPtr->GetTypeName(), yson, requestInfo.Address));
        Timer_.Reset();
    }

    void AfterRequestCallback(const NYP::NClient::TRequestInfo& requestInfo) override {
        Frame_->LogEvent(ELogPriority::TLOG_DEBUG, NLogEvent::TYpClientRequestEnd(requestInfo.Id));
        const ui64 durationTime = Timer_.Get().MicroSeconds();

        if (const auto durationSensor = *Sensors_.FindPtr(DURATION)) {
            durationSensor->Add(durationTime);
        }
    }

private:
    TLogFramePtr Frame_;
    THistogramSensorMap Sensors_;
    TSimpleTimer Timer_;
};

THashMap<TString, THistogramSensorMap> GenerateHistogramSensors(
    const std::initializer_list<TStringBuf>& histogramRequestTypes
    , const std::initializer_list<TStringBuf>& histogramSensorTypes
    , TVector<std::pair<TStringBuf, TStringBuf>> labels
) {
    THashMap<TString, THistogramSensorMap> histogramSensors;
    for (const auto& requestType : histogramRequestTypes) {
        histogramSensors.emplace(requestType, THistogramSensorMap({}));
        labels.emplace_back(REQUEST_TYPE, requestType);
        for (const auto& sensorType: histogramSensorTypes) {
            histogramSensors[requestType].emplace(sensorType, MakeAtomicShared<NInfra::THistogramRateSensor>(
                CTL_SENSOR_GROUP
                , "yp_request." + ToString(sensorType)
                , NMonitoring::ExponentialHistogram(NMonitoring::HISTOGRAM_MAX_BUCKETS_COUNT, BASE_OF_HISTOGRAM, SCALE_OF_HISTOGRAM)
                , labels
            ));
        }
        labels.pop_back();
    }

    return histogramSensors;
}

} // namespace

} // namespace NInfra::NController
