#pragma once

#include <infra/libs/controller/config/config.pb.h>
#include <infra/libs/controller/service/service.h>

#include <infra/libs/sensors/macros.h>
#include <infra/libs/sensors/sensor.h>

#include <library/cpp/proto_config/load.h>

namespace NInfra::NController {

namespace {

const TSensorGroup LOGGER_SENSOR_GROUP("logger");

}

/**
    Will load TControllerConfig from args and resource
*/
template<typename THttpService = TService, typename... TServiceArgs>
int RunDaemon(int argc, const char* argv[], const TString& configResource, TObjectManagersFactoryPtr objectFactory, TServiceArgs&&... serviceArgs) {
    static_assert(std::is_convertible<THttpService*, IService*>::value);

    const TControllerConfig config = NProtoConfig::GetOpt<TControllerConfig>(argc, argv, configResource);
    const NUpdatableProtoConfig::TConfigHolderConfig& updatableConfigOpts = config.GetUpdatableConfigOptions();

    NUpdatableProtoConfig::TConfigHolderPtr<TControllerConfig> configHolder = NUpdatableProtoConfig::CreateConfigHolder(config, updatableConfigOpts);
    NUpdatableProtoConfig::TAccessor<TControllerConfig> controllerConfig = configHolder->Accessor();

    return RunDaemon<THttpService>(controllerConfig, objectFactory, std::forward<TServiceArgs>(serviceArgs)...);
}

template<typename THttpService = TService, typename... TServiceArgs>
int RunDaemon(NUpdatableProtoConfig::TAccessor<TControllerConfig> config, TObjectManagersFactoryPtr objectFactory, TServiceArgs&&... serviceArgs) {
    static_assert(std::is_convertible<THttpService*, IService*>::value);
    return RunDaemon<THttpService>(std::move(config), TVector<TObjectManagersFactoryPtr>{objectFactory}, std::forward<TServiceArgs>(serviceArgs)...);
}

template<typename THttpService = TService, typename... TServiceArgs>
int RunDaemon(NUpdatableProtoConfig::TAccessor<TControllerConfig> config, TVector<TSingleClusterObjectManagersFactoryPtr> singleClusterObjectFactories, TServiceArgs&&... serviceArgs) {
    static_assert(std::is_convertible<THttpService*, IService*>::value);
    TVector<TObjectManagersFactoryPtr> objectFactories(singleClusterObjectFactories.begin(), singleClusterObjectFactories.end());
    return RunDaemon<THttpService>(std::move(config), objectFactories, std::forward<TServiceArgs>(serviceArgs)...);
}

template<typename THttpService = TService, typename... TServiceArgs>
int RunDaemon(NUpdatableProtoConfig::TAccessor<TControllerConfig> config, TVector<TSingleClusterObjectManagerFactoryPtr> singleClusterObjectFactories, TServiceArgs&&... serviceArgs) {
    static_assert(std::is_convertible<THttpService*, IService*>::value);
    TVector<TObjectManagersFactoryPtr> objectFactories(singleClusterObjectFactories.begin(), singleClusterObjectFactories.end());
    return RunDaemon<THttpService>(std::move(config), objectFactories, std::forward<TServiceArgs>(serviceArgs)...);
}

template<typename THttpService = TService, typename... TServiceArgs>
int RunDaemon(NUpdatableProtoConfig::TAccessor<TControllerConfig> controllerConfig, TVector<TObjectManagersFactoryPtr> objectFactories, TServiceArgs&&... serviceArgs) {
    static_assert(std::is_convertible<THttpService*, IService*>::value);
    auto onLogWriteError = [] {
        STATIC_INFRA_RATE_SENSOR(LOGGER_SENSOR_GROUP, "write_failed");
        Cout.Flush();
        Cerr << CurrentExceptionMessage() << Endl;
    };

    const auto controllerConfigSnapshot = controllerConfig.Get();
    TLogger logger(controllerConfigSnapshot->GetLogger(), onLogWriteError);

    TVector<TControllerPtr> controllers;
    {
        THashSet<TString> factoriesNames;
        const auto defaultSyncInterval = FromString<TDuration>(controllerConfigSnapshot->GetController().GetSyncInterval());
        const auto watchObjectsTimeLimit = FromString<TDuration>(controllerConfigSnapshot->GetController().GetWatchObjectsTimeLimit());

        TSet<TString> watchObjectsBannedObjectsTypes(
            controllerConfigSnapshot->GetController().GetWatchObjectsBannedObjectsTypes().begin()
            , controllerConfigSnapshot->GetController().GetWatchObjectsBannedObjectsTypes().end()
        );

        TSet<TString> forceAlwaysSelectObjectsForTypes(
            controllerConfigSnapshot->GetController().GetForceAlwaysSelectObjectsForTypes().begin()
            , controllerConfigSnapshot->GetController().GetForceAlwaysSelectObjectsForTypes().end()
        );

        for (TObjectManagersFactoryPtr objectFactory : objectFactories) {
            Y_ENSURE(factoriesNames.insert(objectFactory->GetFactoryName()).second, "Object factories' names must be unique");
            controllers.push_back(new TControllerBase(objectFactory, defaultSyncInterval, watchObjectsTimeLimit, watchObjectsBannedObjectsTypes, forceAlwaysSelectObjectsForTypes));
        }
    }
    TControllerLoop loop(std::move(controllerConfig), logger, controllers);

    THttpService httpService(controllerConfigSnapshot->GetHttpService(), logger, loop, std::forward<TServiceArgs>(serviceArgs)...);
    httpService.Start(logger.SpawnFrame());

    loop.RunSyncLoop();
    httpService.Wait(logger.SpawnFrame());

    return 0;
}

} // namespace NInfra::NController
