#include "read_only_mode.h"

#include <infra/libs/clients/nanny/nanny_client.h>
#include <infra/libs/clients/yav/yav_client.h>
#include <infra/libs/logger/logger.h>

#include <library/cpp/resource/registry.h>
#include <library/cpp/oauth/oauth.h>

#include <util/system/env.h>
#include <util/stream/fwd.h>

namespace {

void PatchResourcesWithNannyService(NInfra::NController::TReadOnlyModeConfig_TNannyConfig nannyConfig) {
    NInfra::NClients::TNannyClientPtr nannyClient = new NInfra::NClients::TNannyClient(
        new NInfra::NHttpExecuter::THttpExecuter(nannyConfig.GetHttp())
    );

    THashMap<TString, TString> nannyFiles = nannyClient->GetServiceResources()
                                               .SetServiceId(nannyConfig.GetServiceName())
                                               .Execute(NInfra::TLogger({}).SpawnFrame());

    for (const auto& fileMapping : nannyConfig.GetStaticFileMapping()) {
        if (nannyFiles.contains(fileMapping.GetNannyFileName())) {
            TFileOutput localFile(fileMapping.GetLocalFilePath());
            localFile.Write(nannyFiles[fileMapping.GetNannyFileName()]);
        } else {
             ythrow yexception() << "File '" << fileMapping.GetNannyFileName() << "' not found in nanny service '" << nannyConfig.GetServiceName() << "'";
        }
    }
}

void PatchEnvWithYavSecret(NInfra::NController::TReadOnlyModeConfig_TYavConfig yavConfig) {
    NInfra::NClients::TYavClientPtr yavClient = new NInfra::NClients::TYavClient(
        new NInfra::NHttpExecuter::THttpExecuter(yavConfig.GetHttp())
    );

    THashMap<TString, TString> secretVersion = yavClient->GetSecretVersion()
                                                         .SetVersionUuid(yavConfig.GetVersionUuid())
                                                         .Execute(NInfra::TLogger({}).SpawnFrame());

    for (const auto& key : {NInfra::NController::NReadOnlyMode::UNIVERSAL_TOKEN_KEY}) {
        if (!secretVersion.contains(key)) {
            ythrow yexception() << "Mandatory Key '" << key << "' not found in secret version '" << yavConfig.GetVersionUuid() << "'";
        }
    }

    for (const auto& [key, value] : secretVersion) {
        SetEnv(key, value);
    }
}

} // namespace

namespace NInfra::NController::NReadOnlyMode {

void PatchEnvAndResourcesWithReadOnlyModeConfig(TReadOnlyModeConfig readOnlyConfig, TControllerConfig& config) {
    PatchHttpExecuterConfig(
        *(readOnlyConfig.MutableYav()->MutableHttp())
        , GetOauthToken(TOauthTokenParams("4f643a47bc3740dda6a046b955cef480", "27f1435af0de47f9873782a1d8374530"))
    );
    PatchEnvWithYavSecret(readOnlyConfig.GetYav());

    readOnlyConfig.MutableNanny()->MutableHttp()->SetAuthToken(GetEnv(TString(UNIVERSAL_TOKEN_KEY)));
    PatchResourcesWithNannyService(readOnlyConfig.GetNanny());

    config.MutableLeadingInvader()->SetType(NInfra::NLeadingInvader::TConfig_ELeadingInvaderType_MOCK_LEADER);
    config.MutableYpClient()->SetReadOnlyMode(true);
}

void PatchHttpExecuterConfig(NHttpExecuter::THttpExecuterConfig& http, const TString& authToken) {
    http.SetAuthToken(authToken);
    http.SetReadOnlyMode(true);
}

} // namespace NInfra::NController
