#include "object_manager.h"

namespace NInfra::NController {

void ISingleClusterObjectManager::GenerateYpUpdates(
    const TDependentObjects& dependentObjects
    , THashMap<TString, TVector<TRequest>>& requests
    , TLogFramePtr frame
) const {
    GenerateYpUpdates(dependentObjects, requests.begin()->second, frame);
}

TVector<std::pair<TVector<NYP::NClient::TObjectAccessAllowedForSubReq>, TMaybe<TString>>> ISingleClusterObjectManager::GetObjectAccessAllowedForArgumentsWithClusters() const {
    TVector<std::pair<TVector<NYP::NClient::TObjectAccessAllowedForSubReq>, TMaybe<TString>>> result;
    auto subReqs = GetObjectAccessAllowedForArguments();
    result.reserve(subReqs.size());
    for (auto& subReq : subReqs) {
        result.emplace_back(std::move(subReq), Nothing());
    }
    return result;
}

IObjectManagersFactory::IObjectManagersFactory(
    const TString& factoryName
    , TShardPtr shard
    , const bool isResponsibleForLock
    , const TMaybe<TDuration> customSyncInterval
)
    : FactoryName_(shard->BuildFullName(factoryName))
    , IsResponsibleForLock_(isResponsibleForLock)
    , CustomSyncInterval_(customSyncInterval)
    , Shard_(shard)
    , SensorGroup_(FactoryName_)
{
}

TString IObjectManagersFactory::GetFactoryName() const {
    return FactoryName_;
}

bool IObjectManagersFactory::IsResponsibleForLock() const {
    return IsResponsibleForLock_;
}

TMaybe<TVector<TClientConfig>> IObjectManagersFactory::GetYpClientConfigs() const {
    return Nothing();
}

TShardPtr IObjectManagersFactory::GetShard() const {
   return Shard_;
}

const TSensorGroup& IObjectManagersFactory::GetSensorGroupRef() {
    return GetShard()->GetNumberOfShards() == 1 ? CTL_SENSOR_GROUP : SensorGroup_;
}

TMaybe<TVector<TClientConfig>> ISingleClusterObjectManagersFactory::GetYpClientConfigs() const {
     auto clientConfig = GetYpClientConfig();
     if (!clientConfig) {
         return Nothing();
     } else {
         return TVector<TClientConfig>{*clientConfig};
     }
}

TMaybe<TClientConfig> ISingleClusterObjectManagersFactory::GetYpClientConfig() const {
    return Nothing();
}

TMaybe<TVector<TClientConfig>> ISingleClusterObjectManagerFactory::GetYpClientConfigs() const {
    auto clientConfig = GetYpClientConfig();
    if (!clientConfig) {
        return Nothing();
    } else {
        return TVector<TClientConfig>{*clientConfig};
    }
}

TMaybe<TClientConfig> ISingleClusterObjectManagerFactory::GetYpClientConfig() const {
    return Nothing();
}

TVector<TExpected<TObjectManagerPtr, IObjectManagersFactory::TValidationError>> ISingleClusterObjectManagersFactory::GetObjectManagers(
    const TVector<TSelectObjectsResultPtr>& selectorResults
    , TLogFramePtr frame
) const {
    auto singleClusterManagers = GetSingleClusterObjectManagers(selectorResults, frame);
    TVector<TExpected<TObjectManagerPtr, IObjectManagersFactory::TValidationError>> managers;
    managers.reserve(singleClusterManagers.size());
    for (auto& manager : singleClusterManagers) {
        if (manager) {
            managers.emplace_back(manager.Success());
        } else {
            managers.push_back(std::move(manager.Error()));
        }
    }
    return managers;
}

TMaybe<TDuration> IObjectManagersFactory::GetCustomSyncInterval() const {
    return CustomSyncInterval_;
}

TVector<TExpected<TObjectManagerPtr, IObjectManagersFactory::TValidationError>> IObjectManagerFactory::GetObjectManagers(
    const TVector<TSelectObjectsResultPtr>& selectorResults
    , TLogFramePtr frame
) const {
    TVector<TExpected<TObjectManagerPtr, TValidationError>> result;
    for (auto& it : selectorResults[0]->Results) {
        result.emplace_back(GetObjectManager(it, frame));
    }
    return result;
}

TExpected<TObjectManagerPtr, IObjectManagerFactory::TValidationError> ISingleClusterObjectManagerFactory::GetObjectManager(
    const TSelectorResultPtr& selectorResult
    , TLogFramePtr frame
) const {
    auto singleClusterManager = GetSingleClusterObjectManager(selectorResult, frame);
    if (singleClusterManager) {
        return TObjectManagerPtr(singleClusterManager.Success());
    } else {
        return singleClusterManager.Error();
    }
}

TVector<TExpected<TSingleClusterObjectManagerPtr, IObjectManagersFactory::TValidationError>> ISingleClusterObjectManagerFactory::GetSingleClusterObjectManagers(
    const TVector<TSelectObjectsResultPtr>& selectorResults
    , TLogFramePtr frame
) const {
    TVector<TExpected<TSingleClusterObjectManagerPtr, TValidationError>> result;
    result.reserve(selectorResults.size());
    for (auto& it : selectorResults[0]->Results) {
        result.emplace_back(GetSingleClusterObjectManager(it, frame));
    }
    return result;
}

TVector<IObjectManagersFactory::TAggregateArgument> IObjectManagersFactory::GetAggregateArgumentsSafe(NInfra::TLogFramePtr logFrame) const {
    const auto res = GetAggregateArguments(logFrame);
    AggregateResultsRequiredCount_ = res.size();
    return res;
}

TVector<IObjectManager::TSelectArgument> IObjectManagersFactory::GetSelectArgumentsSafe(const TVector<TVector<TSelectorResultPtr>>& aggregateResults, NInfra::TLogFramePtr logFrame) const {
    if (AggregateResultsRequiredCount_) {
        Y_ENSURE(aggregateResults.size() == *AggregateResultsRequiredCount_,
            TStringBuilder() << "Aggregate results count more than expected. Expected: " << *AggregateResultsRequiredCount_ << ". Found: " << aggregateResults.size()); 
    }   
    const auto res = GetSelectArguments(aggregateResults, logFrame);
    SelectorResultsRequiredCount_ = res.size();
    return res;
}

TVector<TExpected<TObjectManagerPtr, IObjectManagersFactory::TValidationError>> IObjectManagersFactory::GetObjectManagersSafe(
    const TVector<TSelectObjectsResultPtr>& selectorResults
    , TLogFramePtr frame
) const {
    Y_ENSURE(selectorResults.size() == *SelectorResultsRequiredCount_,
        TStringBuilder() << "Selector results count more than expected. Expected: " << *SelectorResultsRequiredCount_ << ". Found: " << selectorResults.size());
    return GetObjectManagers(selectorResults, frame);
}

IObjectManagerFactory::IObjectManagerFactory(
    const TString& factoryName
    , TShardPtr shard
    , const bool isResponsibleForLock
    , const TMaybe<TDuration> customSyncInterval
)
    : IObjectManagersFactory(
        factoryName
        , shard
        , isResponsibleForLock
        , customSyncInterval
    )
{
}

TVector<IObjectManager::TSelectArgument> IObjectManagerFactory::GetSelectArguments(const TVector<TVector<TSelectorResultPtr>>& aggregateResults, NInfra::TLogFramePtr logFrame) const {
    return {GetSelectArgument(aggregateResults, logFrame)};
}

#define SELECT_ARGUMENT_TIE(s) s.ObjectType, s.Selector, s.Filter, s.Options, s.ClientFilterOptions.Enabled, s.OverrideYpReqLimitsOptions.Enabled, s.SelectAll, s.ClusterName, s.TotalLimit

bool IObjectManager::TSelectArgument::operator<(const IObjectManager::TSelectArgument& other) const {
    return std::tie(SELECT_ARGUMENT_TIE((*this))) < std::tie(SELECT_ARGUMENT_TIE(other));
}

#undef SELECT_ARGUMENT_TIE

} // namespace NInfra::NController
