#pragma once

#include <infra/libs/controller/config/config.pb.h>

#include <infra/libs/controller/sharding/sharding.h>

#include <yp/cpp/yp/obj_access_allowed_for_req.h>
#include <yp/cpp/yp/request_model.h>
#include <yp/cpp/yp/selector_result.h>
#include <yp/cpp/yp/update_request.h>

#include <infra/libs/logger/log_frame.h>
#include <infra/libs/outcome/result.h>
#include <infra/libs/sensors/sensor.h>

#include <library/cpp/deprecated/atomic/atomic.h>

namespace NInfra::NController {

const TSensorGroup CTL_SENSOR_GROUP("controller");

using TSelectorResultPtr = TAtomicSharedPtr<NYP::NClient::TSelectorResult>;

struct TSelectObjectsResult {
    TVector<TSelectorResultPtr> Results;
    ui64 Timestamp = 0u;
    TString Cluster;

    TSelectObjectsResult() = default;

    TSelectObjectsResult(const TVector<TSelectorResultPtr>& results, ui64 timestamp = 0u, const TString& cluster = "")
        : Results(results)
        , Timestamp(timestamp)
        , Cluster(cluster)
    {
    }
};

struct TGetObjectsResult {
    TVector<TSelectorResultPtr> Results;

    TGetObjectsResult() = default;

    explicit TGetObjectsResult(TVector<TSelectorResultPtr> results)
        : Results(std::move(results))
    {
    }
};

struct TWhitelistData {
    TSet<TString> Selectors;
    TMaybe<TString> KeyField;
    TMaybe<ui32> IndexOfKeyField;
    TSet<TString> WatchSelectors;

    TWhitelistData() = default;

    TWhitelistData(const TClientFilterConfig::TClientFilterWhitelist& whitelist, bool mergeLabels)
    {
        if (whitelist.HasKeyField()) {
            KeyField = whitelist.GetKeyField();
        }

        for (const auto& selector : whitelist.GetSelectors()) {
           Selectors.insert(selector.GetPath());
            if (selector.GetWatchEnabled()) {
                WatchSelectors.insert(selector.GetPath());
            }
        }

        size_t pos = 0;
        size_t skippedFields = 0;
        for (const auto& whitelistedSelector : Selectors) {
            if (mergeLabels && TStringBuf(whitelistedSelector).StartsWith("/labels")) {
                ++skippedFields;
            }
            if (KeyField == whitelistedSelector) {
                if (mergeLabels && skippedFields) {
                    --skippedFields;
                }
                IndexOfKeyField = pos - skippedFields;
                break;
            }
            ++pos;
        }
    }
};

using TSelectObjectsResultPtr = TAtomicSharedPtr<TSelectObjectsResult>;
using TGetObjectsResultPtr = TAtomicSharedPtr<TGetObjectsResult>;

class IObjectManager;
using TObjectManagerPtr = TAtomicSharedPtr<IObjectManager>;

class IObjectManager {
public:
    /* WARNING!
     * All select arguments with empty filter are cached.
     * So many different select arguments with empty filter can cause memory limit.
     */

    struct TClientFilterOptions {
        bool Enabled = false;
        bool WhitelistEnabled = true;
        bool MergeLabels = true;
        bool WatchSelectorsEnabled = false;
        TString AdditionalFilter;
        TMaybe<TWhitelistData> CommonWhitelistData;
        THashMap<TString, TMaybe<TWhitelistData>> WhitelistDataPerCluster;
        TSet<TString> WatchBannedSelectors;

        TClientFilterOptions(TClientFilterConfig config)
            : Enabled(config.GetEnabled())
            , WhitelistEnabled(config.GetWhitelistEnabled())
            , MergeLabels(config.GetMergeLabels())
            , WatchSelectorsEnabled(config.GetWatchSelectorsEnabled())
            , AdditionalFilter(config.GetAdditionalFilter())
        {
            if (!Enabled) {
                return;
            }

            WhitelistDataPerCluster.reserve(config.GetClusterWhitelists().size());

            for (const auto& singleWhitelist : config.GetClusterWhitelists()) {
                WhitelistDataPerCluster[singleWhitelist.GetCluster()] = TWhitelistData(singleWhitelist, MergeLabels);
            }

            if (config.HasCommonWhitelist()) {
                CommonWhitelistData = TWhitelistData(config.GetCommonWhitelist(), MergeLabels);
            }

            for (const auto& bannedSelector : config.GetWatchBannedSelectors()) {
                WatchBannedSelectors.insert(bannedSelector);
            }
        }

        const TMaybe<TWhitelistData>& GetWhitelistData(const TStringBuf cluster) const {
            if (const auto& whitelistDataPtr = WhitelistDataPerCluster.FindPtr(cluster); whitelistDataPtr) {
                return *whitelistDataPtr;
            }
            return CommonWhitelistData;
        }
    };

    struct TOverrideYpReqLimitsOptions {
        bool Enabled = false;
        size_t ReqGetLimit = 0;
        size_t ReqWatchLimit = 0;
        size_t CriticalGetReqSize = 0;
        TOverrideYpReqLimitsOptions(TOverrideYpReqLimitsConfig config)
            : Enabled(config.GetEnabled())
            , ReqGetLimit(config.GetReqGetLimit())
            , ReqWatchLimit(config.GetReqWatchLimit())
            , CriticalGetReqSize(config.GetCriticalGetReqSize())
        {
        }
    };

    struct TSelectArgument {
        // TODO(ismagilas) Documentation for this struct
        /* WARNING!
         * Editing struct members may affect caching select results.
         * Caching depends on < operator from .cpp file
         * Modify very carefully. Make sure no one's caching logics will not be affected!
         */

        NYP::NClient::NApi::NProto::EObjectType ObjectType;
        TVector<TString> Selector;
        TString Filter;
        NYP::NClient::TSelectObjectsOptions Options;
        TClientFilterOptions ClientFilterOptions;
        TOverrideYpReqLimitsOptions OverrideYpReqLimitsOptions;
        bool SelectAll = true;
        TMaybe<TString> ClusterName;
        TMaybe<ui64> TotalLimit;

        TSelectArgument(
            NYP::NClient::NApi::NProto::EObjectType objectType
            , TVector<TString> selector
            , TString filter
            , NYP::NClient::TSelectObjectsOptions options
            , TClientFilterConfig clientFilterConfig = {}
            , TOverrideYpReqLimitsConfig overrideYpReqLimitsConfig = {}
            , bool selectAll = true
            , TMaybe<TString> clusterName = Nothing()
            , TMaybe<ui64> totalLimit = Nothing()
        )
            : ObjectType(std::move(objectType))
            , Selector(std::move(selector))
            , Filter(std::move(filter))
            , Options(std::move(options))
            , ClientFilterOptions(std::move(clientFilterConfig))
            , OverrideYpReqLimitsOptions(std::move(overrideYpReqLimitsConfig))
            , SelectAll(selectAll)
            , ClusterName(std::move(clusterName))
            , TotalLimit(std::move(totalLimit))
        {
        }

        bool operator<(const IObjectManager::TSelectArgument& other) const;
    };

    struct TGetArgument {
        NYP::NClient::NApi::NProto::EObjectType ObjectType;
        TVector<TString> Ids;
        TVector<TString> Selector;

        TGetArgument(
            NYP::NClient::NApi::NProto::EObjectType objectType
            , TVector<TString> ids
            , TVector<TString> selector
        )
            : ObjectType(objectType)
            , Ids(std::move(ids))
            , Selector(std::move(selector))
        {
        }
    };

    struct TDependentObjects {
        const TVector<TSelectObjectsResultPtr> SelectedObjects;
        const TVector<TVector<TVector<TString>>> ObjectsAccessAllowedUserIds;
        const TVector<TGetObjectsResultPtr> GotObjects;
    };

    using TRequest = std::variant<NYP::NClient::TCreateObjectRequest, NYP::NClient::TRemoveObjectRequest, NYP::NClient::TUpdateRequest>;

public:
    virtual ~IObjectManager() = default;

    /*
        ObjectId for logging purposes
    */
    virtual TString GetObjectId() const = 0;

    /*
        @return vector of select requests
        Selected data will be passed to the GenerateYpUpdates function.
        If possible, WatchObjects will be used; for more information see
        IObjectManagersFactory::GetSelectArguments comments.
    */
    virtual TVector<TSelectArgument> GetDependentObjectsSelectArguments() const {
        return {};
    }

    virtual TVector<std::pair<TVector<NYP::NClient::TObjectAccessAllowedForSubReq>, TMaybe<TString>>> GetObjectAccessAllowedForArgumentsWithClusters() const {
        return {};
    }

    virtual TVector<TGetArgument> GetDependentObjectsGetArguments() const {
        return {};
    }

    /*
        Prepare YP updates
        Guarantees:
          * all updates will be made under one transaction
          * order: requests[i] will be made not later than requests[i + 1]
    */
    virtual void GenerateYpUpdates(
        const TDependentObjects& dependentObjects
        , THashMap<TString, TVector<TRequest>>& requests
        , TLogFramePtr frame
    ) const = 0;

    // sets controller's leadereship state
    void SetLeadership(const bool isLeader /* acquiring lock or not */) {
        AtomicSet(AmILeader_, isLeader);
    }

    // true, when controller is acquiring the lock, false - otherwise
    bool AmILeader() const {
        return AtomicGet(AmILeader_);
    }

private:
    TAtomic AmILeader_ = false;
};

class ISingleClusterObjectManager : public IObjectManager {
    using IObjectManager::IObjectManager;
public:
    virtual TVector<TVector<NYP::NClient::TObjectAccessAllowedForSubReq>> GetObjectAccessAllowedForArguments() const {
        return {};
    }

    TVector<std::pair<TVector<NYP::NClient::TObjectAccessAllowedForSubReq>, TMaybe<TString>>> GetObjectAccessAllowedForArgumentsWithClusters() const override final;

    virtual void GenerateYpUpdates(
        const TDependentObjects& dependentObjects
        , TVector<TRequest>& requests
        , TLogFramePtr frame
    ) const = 0;

    void GenerateYpUpdates(
        const TDependentObjects& dependentObjects
        , THashMap<TString, TVector<TRequest>>& requests
        , TLogFramePtr frame
    ) const override final;
};
using TSingleClusterObjectManagerPtr = TAtomicSharedPtr<ISingleClusterObjectManager>;

class IObjectManagersFactory;
using TObjectManagersFactoryPtr = TAtomicSharedPtr<IObjectManagersFactory>;

/**
    base class for reduce-like controllers
*/
class IObjectManagersFactory {
public:
    struct TValidationError {
        NYP::NClient::NApi::NProto::EObjectType ObjectType;
        TString ObjectId;
        TString Message;
    };

    struct TAggregateArgument {
        TString ClusterName;
        NYP::NClient::NApi::NProto::EObjectType ObjectType;
        TVector<TString> GroupByExpressions;
        TVector<TString> AggregateExpressions;
        TString Filter;
    };

public:
    IObjectManagersFactory(
        const TString& factoryName
        , TShardPtr shard
        , const bool isResponsibleForLock = false
        , const TMaybe<TDuration> customSyncInterval = Nothing()
    );

    virtual ~IObjectManagersFactory() = default;

    TString GetFactoryName() const;

    /**
     * true, when factory need controoler's Sync() call, despite the lock is acquired or not
     * false, when factory need controller's Sync() only if lock is acquired
     */
    bool IsResponsibleForLock() const;

    /**
     * YP client config used in corresponding controller.
     * Using common config from TControllerConfig, if returned TMaybe is undefined.
     */
    virtual TMaybe<TVector<TClientConfig>> GetYpClientConfigs() const;

    TVector<TAggregateArgument> GetAggregateArgumentsSafe(NInfra::TLogFramePtr logFrame) const;

    TVector<IObjectManager::TSelectArgument> GetSelectArgumentsSafe(const TVector<TVector<TSelectorResultPtr>>& aggregateResults, NInfra::TLogFramePtr logFrame) const;

    /**
     * Guarantees:
     *   - All YP read-only requests will be made with respect to one, generated in advance, timestamp
     */
    TVector<TExpected<TObjectManagerPtr, TValidationError>> GetObjectManagersSafe(
        const TVector<TSelectObjectsResultPtr>& selectorResults
        , TLogFramePtr frame
    ) const;

    TMaybe<TDuration> GetCustomSyncInterval() const;

    /**
     * Actions to do after TControllerLoop::ControllerSyncLoop finished.
     * Does not guarantee that lock is stil acquired.
     */
    virtual void OnGlobalSyncFinish() {}

    // sets controller's leadereship state
    void SetLeadership(const bool isLeader /* acquiring lock or not */) {
        AtomicSet(AmILeader_, isLeader);
    }

    // true, when controller is acquiring the lock, false - otherwise
    bool AmILeader() const {
        return AtomicGet(AmILeader_);
    }

    TShardPtr GetShard() const;

    const TSensorGroup& GetSensorGroupRef();

protected:
    /*
        @return selectors for yp objects
        Method GetObjectManager will be called for every object.
        If returned select argument's filter is empty, options.FetchTimestamps is false and
        SelectAll_ parameter is true(default), - WatchObjects will be used to sync local data with YP database.
        Using WatchObjects does not affect select results, they will be same as if used SelectObjects.
    */
    virtual TVector<IObjectManager::TSelectArgument> GetSelectArguments(const TVector<TVector<TSelectorResultPtr>>& aggregateResults, NInfra::TLogFramePtr logFrame) const = 0;

    virtual TVector<TAggregateArgument> GetAggregateArguments(NInfra::TLogFramePtr) const {
        return {};
    }

    /*
        @return for every object: TObjectManagerPtr if object valid
    */
    virtual TVector<TExpected<TObjectManagerPtr, TValidationError>> GetObjectManagers(
        const TVector<TSelectObjectsResultPtr>& selectorResults
        , TLogFramePtr frame
    ) const = 0;

private:
    const TString FactoryName_;
    const bool IsResponsibleForLock_;
    const TMaybe<TDuration> CustomSyncInterval_;
    TShardPtr Shard_;
    TSensorGroup SensorGroup_;
    TAtomic AmILeader_ = false;

protected:
    mutable TMaybe<size_t> SelectorResultsRequiredCount_;
    mutable TMaybe<size_t> AggregateResultsRequiredCount_;
};

class ISingleClusterObjectManagersFactory : public IObjectManagersFactory {
    using IObjectManagersFactory::IObjectManagersFactory;
public:
    virtual TMaybe<TClientConfig> GetYpClientConfig() const;

    TMaybe<TVector<TClientConfig>> GetYpClientConfigs() const override final;

    virtual TVector<TExpected<TSingleClusterObjectManagerPtr, TValidationError>> GetSingleClusterObjectManagers(
        const TVector<TSelectObjectsResultPtr>& selectorResults
        , TLogFramePtr frame
    ) const = 0;

    TVector<TExpected<TObjectManagerPtr, IObjectManagersFactory::TValidationError>> GetObjectManagers(
        const TVector<TSelectObjectsResultPtr>& selectorResults
        , TLogFramePtr frame
    ) const override final;
};
using TSingleClusterObjectManagersFactoryPtr = TAtomicSharedPtr<ISingleClusterObjectManagersFactory>;

/**
    base class for map-like controllers
*/
class IObjectManagerFactory: public IObjectManagersFactory {
public:
    virtual TVector<IObjectManager::TSelectArgument> GetSelectArguments(const TVector<TVector<TSelectorResultPtr>>& aggregateResults, NInfra::TLogFramePtr logFrame) const override final;

    virtual TVector<TExpected<TObjectManagerPtr, TValidationError>> GetObjectManagers(
        const TVector<TSelectObjectsResultPtr>& selectorResults
        , TLogFramePtr frame
    ) const override final;

protected:
    IObjectManagerFactory(
        const TString& factoryName
        , TShardPtr shard
        , const bool isResponsibleForLock = false
        , const TMaybe<TDuration> customSyncInterval = Nothing()
    );
    virtual IObjectManager::TSelectArgument GetSelectArgument(const TVector<TVector<TSelectorResultPtr>>& aggregateResults, NInfra::TLogFramePtr logFrame) const = 0;

    /*
        @return TObjectManagerPtr if object valid
    */
    virtual TExpected<TObjectManagerPtr, TValidationError> GetObjectManager(
        const TSelectorResultPtr& selectorResult
        , TLogFramePtr frame
    ) const = 0;
};

class ISingleClusterObjectManagerFactory : public IObjectManagerFactory {
    using IObjectManagerFactory::IObjectManagerFactory;
public:
    virtual TMaybe<TClientConfig> GetYpClientConfig() const;

    TMaybe<TVector<TClientConfig>> GetYpClientConfigs() const override final;

    virtual TExpected<TSingleClusterObjectManagerPtr, TValidationError> GetSingleClusterObjectManager(
        const TSelectorResultPtr& selectorResult
        , TLogFramePtr frame
    ) const = 0;

    TExpected<TObjectManagerPtr, TValidationError> GetObjectManager(
        const TSelectorResultPtr& selectorResult
        , TLogFramePtr frame
    ) const override final;

    TVector<TExpected<TSingleClusterObjectManagerPtr, TValidationError>> GetSingleClusterObjectManagers(
        const TVector<TSelectObjectsResultPtr>& selectorResults
        , TLogFramePtr frame
    ) const;
};
using TSingleClusterObjectManagerFactoryPtr = TAtomicSharedPtr<ISingleClusterObjectManagerFactory>;

} // namespace NInfra::NController
