#include "log_diff_printer.h"

#include <infra/libs/logger/event_iterator.h>
#include <infra/libs/logger/protos/events.ev.pb.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/protobuf/util/pb_io.h>
#include <library/cpp/scheme/scheme.h>

#include <yt/yt/core/yson/protobuf_interop.h>

#include <util/string/builder.h>

namespace NInfra::NController {

namespace {

template <typename TEvent>
bool Is(const NProtoBuf::Message* proto) {
    return TEvent().GetTypeName() == proto->GetTypeName();
}

void JumpToLastServiceStart(TCachedEventIterator& iter) {
    while (iter.SafeNext());

    while (const TEvent* event = iter.SafePrev()) {
        if (Is<NLogEvent::TStartService>(event->GetProto())) {
            return;
        }
    }
}

NLogEvent::EYPObjectType ReflectYpObjectTypeFromString(const TString& literalType) {
    static const auto* protoType = NYT::NYson::ReflectProtobufEnumType(NLogEvent::EYPObjectType_descriptor());
    const auto ypObjectType = NYT::NYson::FindProtobufEnumValueByLiteral<NLogEvent::EYPObjectType>(protoType, literalType);

    Y_ENSURE(
        ypObjectType
        , TStringBuilder() << "Cannot find EYPObjectType corresponding to literal " << literalType
    );

    return *ypObjectType;
}

TSet<NLogEvent::EYPObjectType> ConvertLiteralToProtoObjectTypes(const TVector<TString>& literalTypes) {
    TSet<NLogEvent::EYPObjectType> res;
    for (const auto& type : literalTypes) {
        res.emplace(ReflectYpObjectTypeFromString(type));
    }
    return res;
}

void AddCreateEntryToDiff(NSc::TValue& result, const NLogEvent::TCreateYPObject* event) {
    NSc::TValue entry;
    entry["attributes"] = event->GetAttributes();

    TString objectTypeName = NLogEvent::EYPObjectType_Name(event->GetType());
    result["create"][objectTypeName][event->GetId()] = entry;
}

void AddUpdateEntryToDiff(NSc::TValue& result, const NLogEvent::TUpdateYPObject* event) {
    NSc::TValue entry;
    auto serializeRepeatedPtr = [] (const auto& field) {
        TStringBuilder messages;
        messages << "[";
        for (const auto& msg : field) {

            TStringStream msgStream;
            SerializeToTextFormat(msg, msgStream);
            messages << "`" << msgStream.Str() << "`" << ",";
        }
        messages.pop_back();
        messages << "]";
        return messages;
    };
    entry["set_requests"] = serializeRepeatedPtr(event->GetSetRequests());
    entry["remove_requests"] = serializeRepeatedPtr(event->GetRemoveRequests());

    TString objectTypeName = NLogEvent::EYPObjectType_Name(event->GetType());
    result["update"][objectTypeName][event->GetId()] = entry;
}

void AddRemoveEntryToDiff(NSc::TValue& result, const NLogEvent::TRemoveYPObject* event) {
    TString objectTypeName = NLogEvent::EYPObjectType_Name(event->GetType());
    result["remove"][objectTypeName].Push(event->GetId());
}

void IncFactoryCycles(NSc::TValue& result, const TString& factoryName, ESyncCycleEventType eventType) {
    auto& syncCycleRes = result["sync_cycles"][factoryName];
    if (syncCycleRes.Has(ToString(eventType))) {
        ++syncCycleRes[ToString(eventType)].GetIntNumberMutable();
        return;
    }
    syncCycleRes[ToString(eventType)].SetIntNumber(1);
}

} // namespace

int PrintLogDiff(int argc, const char* argv[]) {
    NLastGetopt::TOpts opts;
    opts.SetFreeArgsNum(1);
    opts.SetFreeArgTitle(0, "<eventlog>", "Event log file");

    TVector<TString> literalObjectTypes;
    opts.AddLongOption('o', "objects", "YP object types to check diff for")
        .Optional()
        .RequiredArgument("EYPObjectType")
        .AppendTo(&literalObjectTypes);

    NLastGetopt::TOptsParseResult parsedOpts(&opts, argc, argv);
    TIteratorOptions options{.FileName = parsedOpts.GetFreeArgs()[0]};
    TCachedEventIterator iter(options);

    const TSet<NLogEvent::EYPObjectType> objectTypes = ConvertLiteralToProtoObjectTypes(literalObjectTypes);

    JumpToLastServiceStart(iter);

    NSc::TValue result;
    result.SetDict();
    bool wasStart = false;
    while (const TEvent* event = iter.SafeNext()) {
        const NProtoBuf::Message* proto = event->GetProto();

        if (Is<NLogEvent::TStartSyncCycle>(proto)) {
            const TString factoryName = dynamic_cast<const NLogEvent::TStartSyncCycle*>(proto)->GetFactoryName();
            IncFactoryCycles(result, factoryName, ESyncCycleEventType::EStart);
            wasStart = true;
        }

        if (Is<NLogEvent::TCreateYPObject>(proto)) {
            auto eventPtr = dynamic_cast<const NLogEvent::TCreateYPObject*>(proto);
            if (objectTypes.contains(eventPtr->GetType())) {
                AddCreateEntryToDiff(result, eventPtr);
            }
        }
        else if (Is<NLogEvent::TUpdateYPObject>(proto)) {
            auto eventPtr = dynamic_cast<const NLogEvent::TUpdateYPObject*>(proto);
            if (objectTypes.contains(eventPtr->GetType())) {
                AddUpdateEntryToDiff(result, eventPtr);
            }
        }
        else if (Is<NLogEvent::TRemoveYPObject>(proto)) {
            auto eventPtr = dynamic_cast<const NLogEvent::TRemoveYPObject*>(proto);
            if (objectTypes.contains(eventPtr->GetType())) {
                AddRemoveEntryToDiff(result, eventPtr);
            }
        } else if (Is<NLogEvent::TSyncCycleSuccess>(proto)) {
            const TString factoryName = dynamic_cast<const NLogEvent::TSyncCycleSuccess*>(proto)->GetFactoryName();
            IncFactoryCycles(result, factoryName, ESyncCycleEventType::ESuccess);
        } else if (Is<NLogEvent::TSyncLoopError>(proto)) {
            const TString factoryName = dynamic_cast<const NLogEvent::TSyncLoopError*>(proto)->GetFactoryName();
            IncFactoryCycles(result, factoryName, ESyncCycleEventType::EError);
        }
    }

    if (!wasStart) {
        Cout << "TStartSyncCycle event was not found for any factory" << Endl;
    }
    Cout << result.ToJson(NSc::TJsonOpts::JO_SORT_KEYS | NSc::TJsonOpts::JO_PARSER_STRICT_UTF8) << Endl;

    return EXIT_SUCCESS;
}

} // namespace NInfra::NServiceController
