#include "service.h"

#include <infra/libs/sensors/sensor_registry.h>

#include <library/cpp/watchdog/watchdog.h>

namespace NInfra::NController::NShardMaster {

TService::TService(const THttpServiceConfig& cfg, TLogger& logger, TMasterLoop& masterLoop)
    : Logger_(logger)
    , MasterLoop_(masterLoop)
    , HttpService_(new THttpService(cfg, CreateRouter(*this)))
{
}

TService::TService(const THttpServiceConfig& cfg, TLogger& logger, TMasterLoop& masterLoop, TRequestRouterPtr router)
    : Logger_(logger)
    , MasterLoop_(masterLoop)
    , HttpService_(new THttpService(cfg, router))
{
}

void TService::Start(TLogFramePtr logFrame) {
    HttpService_->Start(logFrame);
}

void TService::Wait(TLogFramePtr logFrame) {
    HttpService_->Wait(logFrame);
}

void TService::Ping(TRequestPtr<TReqPing>, TReplyPtr<TRspPing> reply) {
    TRspPing rsp;
    rsp.SetData("pong");
    reply->Set(rsp);
}

void TService::Leader(TRequestPtr<TReqLeader>, TReplyPtr<TRspLeader> reply) {
    TRspLeader rsp;
    auto leaderInfo = MasterLoop_.GetLeaderInfo();
    if (!leaderInfo.Fqdn.empty()) {
        rsp.SetFqdn(leaderInfo.Fqdn);
    }
    if (!leaderInfo.Ip.empty()) {
        rsp.SetIp(leaderInfo.Ip);
    }

    switch (leaderInfo.LeaderStatus) {
        case NLeadingInvader::TLeaderInfo::EResolveLeaderStatus::SUCCEED:
            rsp.SetLeaderStatus(TRspLeader_EResolveLeaderStatus_SUCCEED);
            break;
        case NLeadingInvader::TLeaderInfo::EResolveLeaderStatus::FAILED:
            rsp.SetLeaderStatus(TRspLeader_EResolveLeaderStatus_FAILED);
            break;
        case NLeadingInvader::TLeaderInfo::EResolveLeaderStatus::NO_LEADER:
            rsp.SetLeaderStatus(TRspLeader_EResolveLeaderStatus_NO_LEADER);
            break;
    }
    reply->Set(rsp);
}

void TService::Shutdown(TRequestPtr<TReqShutdown>, TReplyPtr<TRspShutdown>) {
    static THolder<IWatchDog> abortWatchDog = THolder<IWatchDog>(CreateAbortByTimeoutWatchDog(CreateAbortByTimeoutWatchDogOptions(TDuration::Minutes(3)), "Ooops!"));

    HttpService_->ShutDown();
    MasterLoop_.Shutdown();
}

void TService::Sensors(TRequestPtr<TReqSensors>, TReplyPtr<TRspSensors> reply) {
    TStringStream out;
    SensorRegistryPrint(out);

    TRspSensors result;
    result.SetData(out.Str());

    reply->Set(result);
}

void TService::ReopenLog(TRequestPtr<TReqReopenLog>, TReplyPtr<TRspReopenLog>) {
    Logger_.SpawnFrame()->LogEvent(NLogEvent::TReopenLog());
    Logger_.ReopenLog();
    MasterLoop_.ReopenLogs();
}

} // namespace NInfra::NController::NShardMaster

