#pragma once

#include <infra/libs/controller/shard_master/config/config.pb.h>
#include <infra/libs/controller/shard_master/sensors/sensors.h>

#include <infra/libs/leading_invader/leading_invader.h>
#include <infra/libs/logger/logger.h>
#include <infra/libs/sensors/sensor.h>
#include <infra/libs/updatable_proto_config/accessor.h>

#include <mapreduce/yt/interface/client.h>

#include <util/system/rwlock.h>
#include <util/thread/pool.h>

namespace NInfra::NController::NShardMaster {

namespace {

TSensorGroup NamedSensorGroup(const TSensorGroup& group, const TString& nameLabel) {
    TSensorGroup result = group;
    result.AddLabel(NSensors::SERVICE_NAME, nameLabel);
    return result;
}

} // namespace

class TDistributionManager {
public:
    TDistributionManager(
        NUpdatableProtoConfig::TAccessor<TDistributionManagerConfig> config
    )
        : Config_(std::move(config))
        , ActualConfig_(*Config_)
        , ServiceName_(ActualConfig_.GetServiceName())
        , LeadingInvaderConfig_(ActualConfig_.GetLeadingInvader())
        , YtClient_(
            NYT::CreateClient(
                LeadingInvaderConfig_.GetProxy(),
                NYT::TCreateClientOptions()
                    .Token(LeadingInvaderConfig_.GetToken())
            )
        )
        , SensorGroup_(NamedSensorGroup(NSensors::CTL_SHARD_MASTER_SENSOR_GROUP, GetName()))
        , ManagementDurationSensor_(SensorGroup_, NSensors::MANAGE_SHARDS_DISTRIBUTION_TIME)
    {
        Config_.SubscribeForUpdate([this](
            const TDistributionManagerConfig& oldConfig,
            const TDistributionManagerConfig& newConfig,
            const NUpdatableProtoConfig::TWatchContext& context = {})
        {
            if (context.Id == GetName() && !google::protobuf::util::MessageDifferencer::Equivalent(oldConfig, newConfig)) {
                ActualConfig_ = newConfig;
            }
        });
    }

public:
    TString GetLeadingInvaderName() const;

    TExpected<void, NLeadingInvader::TError> EnsureLeading();

    NLeadingInvader::TLeaderInfo GetLeaderInfo() const;

    void ResetLeadingInvader(
        const std::function<void()>& onLockAcquired = []{}
        , const std::function<void()>& onLockLost = []{}
    );

    void DestroyLeadingInvader();

public:
    const TSensorGroup& GetSensorGroupRef();

    void OnGlobalManagementFinish();

    TDuration GetManagementInterval() const;

    TString GetName() const;

    void IncrementSensor(const TStringBuf sensor, ui64 x);

    void ManageShardsDistribution(
        TLogFramePtr frame
    );

private:
    THashMap<size_t, THashSet<TString>> DiscoverAliveHostsPerShard(TLogFramePtr frame);

    THashMap<size_t, TString> GetShardsCurrentHosts(TLogFramePtr frame);

    THashMap<TString, TVector<size_t>> GetListOfShardsPerHost(
        THashMap<size_t, TString> shardId2CurrentHost,
        TLogFramePtr frame
    );

    void RebalanceShards(
        THashMap<size_t, THashSet<TString>>& shardId2AliveHosts,
        THashMap<size_t, TString>& shardId2CurrentHost,
        THashMap<TString, TVector<size_t>>& host2ListOfShards,
        TLogFramePtr frame
    );

    void MoveShard(
        size_t shardId,
        const TString& oldHost,
        const TString& newHost,
        TLogFramePtr frame
    );

private:
    NUpdatableProtoConfig::TAccessor<TDistributionManagerConfig> Config_;
    TDistributionManagerConfig ActualConfig_;
    const TString ServiceName_;
    NLeadingInvader::TConfig LeadingInvaderConfig_;
    NYT::IClientPtr YtClient_;
    NLeadingInvader::TLeadingInvaderHolder LeadingInvader_;
    TRWMutex LeadingInvaderMutex_;
    TSensorGroup SensorGroup_;
    TDurationSensor ManagementDurationSensor_;
};

using TDistributionManagerPtr = TAtomicSharedPtr<TDistributionManager>;

} // namespace NInfra::NController::NShardMaster
