#include "master_loop.h"

#include <infra/libs/sensors/macros.h>
#include <infra/libs/memory_lock/memory_lock.h>

#include <util/thread/factory.h>

namespace NInfra::NController::NShardMaster {

namespace {

////////////////////////////////////////////////////////////////////////////////

struct TThreadData {
    TThreadData(
        TMasterLoop* masterLoop
        , TDistributionManagerPtr distributionManager
        , NUpdatableProtoConfig::TAccessor<TConfig> config
    )
        : MasterLoop(masterLoop)
        , DistributionManager(distributionManager)
        , Config(std::move(config))
        , ActualConfig(*Config)
    {
        Config.SubscribeForUpdate([this](
            const TConfig& oldConfig,
            const TConfig& newConfig,
            const NUpdatableProtoConfig::TWatchContext& context = {}
        ) {
            if (context.Id == DistributionManager->GetName() && !google::protobuf::util::MessageDifferencer::Equivalent(oldConfig, newConfig)) {
                ActualConfig = newConfig;
            }
        });
    }

    TMasterLoop* MasterLoop;
    TDistributionManagerPtr DistributionManager;
    NUpdatableProtoConfig::TAccessor<TConfig> Config;
    TConfig ActualConfig;
};

////////////////////////////////////////////////////////////////////////////////

} // namespace

TMasterLoop::TMasterLoop(
    NUpdatableProtoConfig::TAccessor<TConfig> config,
    TLogger& logger,
    TVector<TDistributionManagerPtr> distributionManagers
)
    : Config_(std::move(config))
    , Logger_(logger)
    , DistributionManagers_(distributionManagers)
{
    const auto initialConfig = Config_.Get();
    NMemoryLock::LockSelfMemory(initialConfig->GetMemoryLock(), Logger_.SpawnFrame(), NSensors::CTL_SHARD_MASTER_SENSOR_GROUP);
    AtomicSet(Running_, 1);
}

////////////////////////////////////////////////////////////////////////////////

void TMasterLoop::ReopenLogs() {
    Config_.RequestReopenLogs();
}

NLeadingInvader::TLeaderInfo TMasterLoop::GetLeaderInfo(TMaybe<TString> leadingInvaderName) const {
    if (!leadingInvaderName.Defined()) {
        if (DistributionManagers_.size() != 1) {
            return NLeadingInvader::TLeaderInfo{NLeadingInvader::TLeaderInfo::EResolveLeaderStatus::FAILED, "", ""};
        }

        leadingInvaderName = DistributionManagers_[0]->GetLeadingInvaderName();
    }

    for (const auto& distributionManager : DistributionManagers_) {
        if (distributionManager->GetLeadingInvaderName() == leadingInvaderName.GetRef()) {
            return distributionManager->GetLeaderInfo();
        }
    }

    return NLeadingInvader::TLeaderInfo{NLeadingInvader::TLeaderInfo::EResolveLeaderStatus::FAILED, "", ""};
}

void TMasterLoop::Shutdown() {
    AtomicSet(Running_, 0);
}

////////////////////////////////////////////////////////////////////////////////

void TMasterLoop::RunLoop() {
    Config_.RequestUpdate(NUpdatableProtoConfig::TWatchContext());

    TVector<TThreadData> distributionManagerThreadsData;
    distributionManagerThreadsData.reserve(DistributionManagers_.size());
    for (TDistributionManagerPtr distributionManager : DistributionManagers_) {
        distributionManagerThreadsData.emplace_back(this, distributionManager, Config_);
    }

    TVector<THolder<IThreadFactory::IThread>> distributionManagerThreads(distributionManagerThreadsData.size());
    for (size_t i = 0; i < distributionManagerThreadsData.size(); ++i) {
        distributionManagerThreads[i] = SystemThreadFactory()->Run([&distributionManagerThreadsData, i](){
            RunDistributionManagerLoop((void*)&distributionManagerThreadsData[i]);
        });
    }

    for (const auto&  thread : distributionManagerThreads) {
        thread->Join();
    }

    for (TDistributionManagerPtr distributionManager : DistributionManagers_) {
        distributionManager->OnGlobalManagementFinish();
    }
}

void* TMasterLoop::RunDistributionManagerLoop(
    void* data
) {
    TThreadData* threadData = (TThreadData*)data;

    TGetActualConfig getActualConfig = [&]() -> std::pair<const TConfig&, bool> {
        bool hasUpdate = threadData->Config.RequestUpdate({threadData->DistributionManager->GetName()});
        return {threadData->ActualConfig, hasUpdate};
    };

    threadData->MasterLoop->DistributionManagerLoop(
        threadData->DistributionManager,
        std::move(getActualConfig)
    );
    return nullptr;
}

void TMasterLoop::DistributionManagerLoop(
    TDistributionManagerPtr distributionManager
    , TGetActualConfig getActualConfig
) {
    const auto onLockAcquired = [&distributionManager]() {
        NON_STATIC_INFRA_INT_GAUGE_SENSOR(distributionManager->GetSensorGroupRef(), NSensors::LOCK_ACQUIRED, 1);
    };

    const auto onLockLost = [&distributionManager]() {
        NON_STATIC_INFRA_INT_GAUGE_SENSOR(distributionManager->GetSensorGroupRef(), NSensors::LOCK_ACQUIRED, 0);
    };

    while (AtomicGet(Running_) == 1) {
        distributionManager->ResetLeadingInvader(onLockAcquired, onLockLost);

        while (AtomicGet(Running_) == 1) {
            auto [actualConfig, _] = getActualConfig();
            auto frame = Logger_.SpawnFrame();

            if (auto result = distributionManager->EnsureLeading(); !(bool)result) {
                frame->LogEvent(
                    ELogPriority::TLOG_WARNING
                    , NLogEvent::TLockAcquireError(distributionManager->GetLeadingInvaderName(), result.Error().Reason)
                );

                Sleep(FromString<TDuration>(actualConfig.GetMasterLoopConfig().GetFollowerLoopInterval()));
                continue;
            } else {
                frame->LogEvent(NLogEvent::TLockAcquireSuccess(distributionManager->GetLeadingInvaderName()));
            }

            try {
                RunManageShardsDistribution(distributionManager, actualConfig);
            } catch (...) {
                break;
            }

            frame->LogEvent(
                ELogPriority::TLOG_DEBUG
                , NLogEvent::TSleep(
                    "Regular"
                    , ToString(distributionManager->GetManagementInterval())
                    , distributionManager->GetName()
                )
            );
            Sleep(distributionManager->GetManagementInterval());
        }

        // Service has been shutdown or ManageShardsDistribution ended with error
        distributionManager->DestroyLeadingInvader();
    }
}

void TMasterLoop::RunManageShardsDistribution(
    TDistributionManagerPtr distributionManager
    , const TConfig& actualConfig
) {
    Y_UNUSED(actualConfig);

    auto frame = Logger_.SpawnFrame();

    distributionManager->IncrementSensor(NSensors::RESHARDING_CYCLES, 1);
    const TInstant startManageShardsDistribution = TInstant::Now();

    try {
        distributionManager->ManageShardsDistribution(frame);
    } catch (...) {
        distributionManager->IncrementSensor(NSensors::FAILED_RESHARDING_CYCLES, 1);
        frame->LogEvent(
            ELogPriority::TLOG_ERR
            , NLogEvent::TManageShardsDistributionError(distributionManager->GetName(), CurrentExceptionMessage())
        );
        throw;
    }

    distributionManager->IncrementSensor(NSensors::SUCCESSFUL_RESHARDING_CYCLES, 1);

    const auto manageShardsDistributionTime = TInstant::Now() - startManageShardsDistribution;
    frame->LogEvent(NLogEvent::TManageShardsDistributionSuccess(ToString(manageShardsDistributionTime), distributionManager->GetName()));
}

} // namespace NInfra::NController::NShardMaster
