#include "sharding.h"

#include <infra/libs/yp_dns/dynamic_zones/helpers/yt/yt.h>

#include <mapreduce/yt/util/ypath_join.h>

#include <library/cpp/retry/retry.h>

#include <util/system/hostname.h>

namespace NInfra::NController {

////////////////////////////////////////////////////////////////////////////////

TString TShardMasterDistributionSubscriber::BuildLivenessNodePath(const TString& prefix, size_t shardId) {
    const TString shardPart = TStringBuilder() << "shard_id_" << shardId;
    return NYT::JoinYPaths(prefix, shardPart, "alive_hosts", HostName());
}

TString TShardMasterDistributionSubscriber::BuildGivenJobNodePath(const TString& prefix, size_t shardId) {
    const TString shardPart = TStringBuilder() << "shard_id_" << shardId;
    return NYT::JoinYPaths(prefix, shardPart, "given_job");
}

TShardMasterDistributionSubscriber::TShardMasterDistributionSubscriber(
    bool managedByMaster
    , size_t shardId
    , const TString& ensureMasterLivenessTime
    , const TString& masterLeadingInvaderPath
    , const NLeadingInvader::TConfig& livenessLeadingInvaderConfig)
    : ManagedByMaster_(managedByMaster)
    , ShardId_(shardId)
    , LivenessLeadingInvaderConfig_(livenessLeadingInvaderConfig)
    , YtFolderPrefix_(LivenessLeadingInvaderConfig_.GetPath())
    , EnsureMasterLivenessTime_(ensureMasterLivenessTime)
    , MasterLeadingInvaderPath_(masterLeadingInvaderPath)
    , GivenJobNodePath_(BuildGivenJobNodePath(YtFolderPrefix_, ShardId_))
    , YtClient_(NYT::CreateClient(
        LivenessLeadingInvaderConfig_.GetProxy(),
        NYT::TCreateClientOptions()
            .Token(LivenessLeadingInvaderConfig_.GetToken())))
{
    LivenessLeadingInvaderConfig_.SetPath(TShardMasterDistributionSubscriber::BuildLivenessNodePath(YtFolderPrefix_, ShardId_));
}

bool TShardMasterDistributionSubscriber::RegisterLiveness(
    TLogFramePtr frame
    , const std::function<void()>& onLockAcquired
    , const std::function<void()>& onLockLost
) {
    Y_ENSURE(ManagedByMaster_);
    frame->LogEvent(NLogEvent::TRegisterLivenessStart(LivenessLeadingInvaderConfig_.GetPath()));

    try {
        DoWithRetry<std::exception>(
            [this, &onLockAcquired, &onLockLost, &frame] {
                NYpDns::NYtHelpers::CreateCypressNodeIfMissing(
                    YtClient_,
                    LivenessLeadingInvaderConfig_.GetPath(),
                    NYT::ENodeType::NT_STRING,
                    frame);
                LivenessLeadingInvader_.Reset(NLeadingInvader::CreateLeadingInvader(LivenessLeadingInvaderConfig_, onLockAcquired, onLockLost));
                frame->LogEvent(NLogEvent::TRegisterLivenessSuccess(LivenessLeadingInvaderConfig_.GetPath()));
            },
            [this, &frame] (const std::exception& ex) {
                frame->LogEvent(TLOG_WARNING, NLogEvent::TRegisterLivenessError(LivenessLeadingInvaderConfig_.GetPath(), ex.what()));
            },
            TRetryOptions().WithCount(3).WithSleep(TDuration::Seconds(1)).WithIncrement(TDuration::Seconds(1)),
            /* throwLast */ true
        );

        return true;
    } catch (...) {
        frame->LogEvent(TLOG_ERR, NLogEvent::TRegisterLivenessFail(LivenessLeadingInvaderConfig_.GetPath(), CurrentExceptionMessage()));
        return false;
    }
}

bool TShardMasterDistributionSubscriber::EnsureTaskPermissionByMaster(TLogFramePtr frame) {
    Y_ENSURE(ManagedByMaster_);
    frame->LogEvent(NLogEvent::TEnsureTaskPermissionByMasterStart(GivenJobNodePath_));

    try {
        auto result = DoWithRetry<bool, std::exception>(
            [this, &frame] {
                NYpDns::NYtHelpers::CreateCypressNodeIfMissing(
                    YtClient_,
                    GivenJobNodePath_,
                    NYT::ENodeType::NT_STRING,
                    frame);

                auto result = NYpDns::NYtHelpers::TryGetNodeData(YtClient_, GivenJobNodePath_, frame).AsString() == HostName();
                frame->LogEvent(NLogEvent::TEnsureTaskPermissionByMasterSuccess(GivenJobNodePath_, result));
                return result;
            },
            [this, &frame] (const std::exception& ex) {
                frame->LogEvent(TLOG_WARNING, NLogEvent::TEnsureTaskPermissionByMasterError(GivenJobNodePath_, ex.what()));
            },
            TRetryOptions().WithCount(3).WithSleep(TDuration::Seconds(1)).WithIncrement(TDuration::Seconds(1)),
            /* throwLast */ true
        );
        Y_ENSURE(result.Defined());

        return *result;
    } catch (...) {
        frame->LogEvent(TLOG_ERR, NLogEvent::TEnsureTaskPermissionByMasterFail(GivenJobNodePath_, CurrentExceptionMessage()));
        return false;
    }
}

bool TShardMasterDistributionSubscriber::EnsureShardMasterIsAlive(TLogFramePtr frame) {
    //TODO(ismagilas) Rewrite with DoWithRetry, this looks horrible
    Y_ENSURE(ManagedByMaster_);

    frame->LogEvent(NLogEvent::TEnsureShardMasterIsAliveStart(MasterLeadingInvaderPath_));
    const TInstant maxAllowedTimePoint = TInstant::Now() + FromString<TDuration>(EnsureMasterLivenessTime_);
    while (true) {
        try {
            if (NYpDns::NYtHelpers::IsNodeLocked(YtClient_, MasterLeadingInvaderPath_, frame)) {
                frame->LogEvent(NLogEvent::TShardMasterIsAlive(MasterLeadingInvaderPath_));
                return true;
            }
        } catch (...) {
            frame->LogEvent(TLOG_ERR, NLogEvent::TEnsureShardMasterIsAliveError(MasterLeadingInvaderPath_, CurrentExceptionMessage()));
        }

        Sleep(TDuration::Seconds(1));

        if (maxAllowedTimePoint < TInstant::Now()) {
            frame->LogEvent(TLOG_ERR, NLogEvent::TNoAliveShardMasterFound(MasterLeadingInvaderPath_));
            return false;
        }
    }
}

////////////////////////////////////////////////////////////////////////////////

TShard::TShard(
    size_t numberOfShards
    , size_t id
    , bool managedByMaster
    , const TString& ensureMasterLivenessTime
    , const TString& masterLeadingInvaderPath
    , const NLeadingInvader::TConfig& leadingInvaderConfig
    , const NLeadingInvader::TConfig& livenessLeadingInvaderConfig)
    : NumberOfShards_(numberOfShards)
    , Id_(id)
    , ManagedByMaster_(managedByMaster)
    , EnsureMasterLivenessTime_(ensureMasterLivenessTime)
    , MasterLeadingInvaderPath_(masterLeadingInvaderPath)
    , LeadingInvaderConfig_(leadingInvaderConfig)
    , ShardMasterDistributionSubscriber_(MakeAtomicShared<TShardMasterDistributionSubscriber>(
        ManagedByMaster_
        , Id_
        , EnsureMasterLivenessTime_
        , MasterLeadingInvaderPath_
        , livenessLeadingInvaderConfig
    ))
{
    LeadingInvaderConfig_.SetPath(BuildFullName(LeadingInvaderConfig_.GetPath()));
}

size_t TShard::GetNumberOfShards() const {
    return NumberOfShards_;
}

size_t TShard::GetShardId() const {
    return Id_;
}

bool TShard::IsManagedByMaster() const {
    return ManagedByMaster_;
}

NLeadingInvader::TConfig TShard::GetLeadingInvaderConfig() const {
    return LeadingInvaderConfig_;
}

TShardMasterDistributionSubscriberPtr TShard::GetShardMasterDistributionSubscriber() const {
    return ShardMasterDistributionSubscriber_;
}

TString TShard::GetFullLeadingInvaderName() const {
    return LeadingInvaderConfig_.GetPath();
}

TString TShard::BuildFullName(const TString& name) const {
    return NumberOfShards_ == 1
        ? name
        : TStringBuilder() << name << "_shard_id_" << Id_;
}

TExpected<void, NLeadingInvader::TError> TShard::EnsureLeading() {
    TReadGuard guard(LeadingInvaderMutex_);
    if (!LeadingInvader_) {
        return NLeadingInvader::TError{TString("LeadingInvader has been destroyed")};
    }
    return LeadingInvader_->EnsureLeading();
}

NLeadingInvader::TLeaderInfo TShard::GetLeaderInfo() const {
    TReadGuard guard(LeadingInvaderMutex_);
    if (!LeadingInvader_) {
        return NLeadingInvader::TLeaderInfo{NLeadingInvader::TLeaderInfo::EResolveLeaderStatus::FAILED, "", ""};
    }
    return LeadingInvader_->GetLeaderInfo();
}

void TShard::ResetLeadingInvader(
    const std::function<void()>& onLockAcquired
    , const std::function<void()>& onLockLost
    , bool ignoreIfSet
) {
    TWriteGuard guard(LeadingInvaderMutex_);
    if (ignoreIfSet && LeadingInvader_) {
        return;
    }
    LeadingInvader_.Reset(NLeadingInvader::CreateLeadingInvader(LeadingInvaderConfig_, onLockAcquired, onLockLost));
}

void TShard::DestroyLeadingInvader() {
    TWriteGuard guard(LeadingInvaderMutex_);
    LeadingInvader_.Destroy();
}

////////////////////////////////////////////////////////////////////////////////

TSharding::TSharding()
    : NumberOfShards_(1)
{
    Shards_.reserve(NumberOfShards_);
    for (size_t i = 0; i < NumberOfShards_; ++i) {
        Shards_.emplace_back(MakeAtomicShared<TShard>(
            NumberOfShards_,
            i,
            ManagedByMaster_,
            EnsureMasterLivenessTime_,
            MasterLeadingInvaderPath_,
            LeadingInvaderConfig_,
            LivenessLeadingInvaderConfig_
        ));
    }
}

TSharding::TSharding(
    const NLeadingInvader::TConfig& leadingInvaderConfig
    , const NLeadingInvader::TConfig& livenessLeadingInvaderConfig
    , size_t numberOfShards
    , bool managedByMaster
    , const TString& ensureMasterLivenessTime
    , const TString& masterLeadingInvaderPath)
    : NumberOfShards_(numberOfShards)
    , ManagedByMaster_(managedByMaster)
    , EnsureMasterLivenessTime_(ensureMasterLivenessTime)
    , MasterLeadingInvaderPath_(masterLeadingInvaderPath)
    , LeadingInvaderConfig_(leadingInvaderConfig)
    , LivenessLeadingInvaderConfig_(livenessLeadingInvaderConfig)
{
    Shards_.reserve(NumberOfShards_);
    for (size_t i = 0; i < NumberOfShards_; ++i) {
        Shards_.emplace_back(MakeAtomicShared<TShard>(
            NumberOfShards_,
            i,
            ManagedByMaster_,
            EnsureMasterLivenessTime_,
            MasterLeadingInvaderPath_,
            LeadingInvaderConfig_,
            LivenessLeadingInvaderConfig_
        ));
    }
}

TShardPtr TSharding::GetShard(size_t id) {
    return Shards_[id];
}

size_t TSharding::GetNumberOfShards() const {
    return NumberOfShards_;
}

NLeadingInvader::TConfig TSharding::GetLeadingInvaderConfig() const {
    return LeadingInvaderConfig_;
}

NLeadingInvader::TConfig TSharding::GetLivenessLeadingInvaderConfig() const {
    return LivenessLeadingInvaderConfig_;
}

////////////////////////////////////////////////////////////////////////////////

} // namespace NInfra::NController
