#include "standalone_controller.h"

#include <yp/cpp/yp/token.h>

#include <library/cpp/proto_config/config.h>

namespace NInfra::NController {

TStandaloneController::TStandaloneController(NInfra::NController::TControllerConfig config, TObjectManagersFactoryPtr objectFactory)
    : TStandaloneController(std::move(config), TVector<TObjectManagersFactoryPtr>{objectFactory})
{}

TStandaloneController::TStandaloneController(NInfra::NController::TControllerConfig config, TVector<TSingleClusterObjectManagersFactoryPtr> objectFactories)
    : TStandaloneController(std::move(config), TVector<TObjectManagersFactoryPtr>(objectFactories.begin(), objectFactories.end()))
{}

TStandaloneController::TStandaloneController(NInfra::NController::TControllerConfig config, TVector<TSingleClusterObjectManagerFactoryPtr> objectFactories)
    : TStandaloneController(std::move(config), TVector<TObjectManagersFactoryPtr>(objectFactories.begin(), objectFactories.end()))
{}

TStandaloneController::TStandaloneController(
    NInfra::NController::TControllerConfig config
    , TVector<TObjectManagersFactoryPtr> objectFactories
)
    : Logger_(config.GetLogger())
    , Config_(std::move(config))
    , ObjectFactories_(objectFactories)
    , RetryCount_(Config_.GetController().GetRetryCount())
    , ThreadPoolSize_(Config_.GetController().GetThreadPoolSize())
    , AuxThreadPoolSize_(Config_.GetController().GetAuxThreadPoolSize())
    , MtpQueue_(TThreadPool())
    , YpRequestsBatchSize_(Config_.GetController().GetYpRequestsBatchSize())
    , MaxSeqFailedObjMngrsCount_(Config_.GetController().GetMaxSequentialFailedObjMngrsCount())
    , DefaultSyncInterval_(FromString<TDuration>(Config_.GetController().GetSyncInterval()))
    , WatchObjectsTimeLimit_(FromString<TDuration>(Config_.GetController().GetWatchObjectsTimeLimit()))
    , WatchObjectsBannedObjectsTypes_(
        Config_.GetController().GetWatchObjectsBannedObjectsTypes().begin()
        , Config_.GetController().GetWatchObjectsBannedObjectsTypes().end()
    )
    , ForceAlwaysSelectObjectsForTypes_(
        Config_.GetController().GetForceAlwaysSelectObjectsForTypes().begin()
        , Config_.GetController().GetForceAlwaysSelectObjectsForTypes().end()
    )
{
    if (AuxThreadPoolSize_ == 0) {
        AuxMtpQueue_.Reset(new TFakeThreadPool());
    } else {
        AuxMtpQueue_.Reset(new TThreadPool());
    }

    Controllers_.reserve(ObjectFactories_.size());
    THashSet<TString> factoriesNames;
    for (TObjectManagersFactoryPtr objectFactory : ObjectFactories_) {
        Y_ENSURE(factoriesNames.insert(objectFactory->GetFactoryName()).second, "Object factories' names must be unique");
        Controllers_.push_back(new TControllerBase(objectFactory, DefaultSyncInterval_, WatchObjectsTimeLimit_, std::move(WatchObjectsBannedObjectsTypes_), std::move(ForceAlwaysSelectObjectsForTypes_)));
    }
}

TStandaloneController::TStandaloneController(TStringBuf config, TObjectManagersFactoryPtr objectFactory)
    : TStandaloneController(NProtoConfig::ParseConfigFromJson<NInfra::NController::TControllerConfig>(config), objectFactory)
{}

void TStandaloneController::Sync() {
    MtpQueue_.Start(ThreadPoolSize_ + ObjectFactories_.size());
    AuxMtpQueue_->Start(AuxThreadPoolSize_);

    TVector<NThreading::TFuture<void>> prepareInvadersFuture(ObjectFactories_.size());
    for (size_t i = 0; i < ObjectFactories_.size(); ++i) {
        TObjectManagersFactoryPtr& objectFactory = ObjectFactories_[i];
        prepareInvadersFuture[i] = NThreading::Async(
            [&objectFactory]() {
                objectFactory->GetShard()->ResetLeadingInvader();
            }
            , MtpQueue_
        );
    }

    for (size_t i = 0; i < prepareInvadersFuture.size(); ++i) {
        prepareInvadersFuture[i].GetValueSync();
    }

    TVector<NThreading::TFuture<void>> controllerFuture(ObjectFactories_.size());
    for (size_t i = 0; i < Controllers_.size(); ++i) {
        TObjectManagersFactoryPtr& objectFactory = ObjectFactories_[i];
        TControllerPtr& controller = Controllers_[i];
        controllerFuture[i] = NThreading::Async(
            [this, &objectFactory, &controller]() {
                {
                    while (/*!objectFactory->IsResponsibleForLock() && */!(bool)objectFactory->GetShard()->EnsureLeading()) {
                        continue;
                    }
                }
                TVector<NInfra::NController::TClientConfig> configs;
                if (auto customConfigs = controller->GetYpClientConfigs(); customConfigs) {
                    configs = std::move(*customConfigs);
                } else if (Config_.HasYpClient()) {
                    configs.push_back(Config_.GetYpClient());
                } else {
                    configs.reserve(Config_.GetYpClients().ConfigsSize());
                    for (const auto& ypClientConfig : Config_.GetYpClients().GetConfigs()) {
                        configs.push_back(ypClientConfig);
                    }
                }
                Y_ENSURE(!configs.empty());

                THashMap<TString, NYP::NClient::TClientPtr> auxClients;
                THashMap<TString, NYP::NClient::TClientPtr> clients;
                THashMap<TString, NYP::NClient::TTransactionFactoryPtr> transactionFactories;

                for (const auto& ypClientConfig : configs) {
                    const TString clusterName = ypClientConfig.GetClusterName();

                    NYP::NClient::TClientOptions ypOpts;
                    ypOpts
                        .SetAddress(ypClientConfig.GetAddress())
                        .SetEnableSsl(ypClientConfig.GetEnableSsl())
                        .SetTimeout(TDuration::Parse(ypClientConfig.GetTimeout()))
                        .SetReadOnlyMode(ypClientConfig.GetReadOnlyMode())
                        .SetToken(NYP::NClient::FindToken())
                        .SetSnapshotTimestamp(ypClientConfig.GetSnapshotTimestamp())
                        .SetMaxReceiveMessageSize(ypClientConfig.GetMaxReceiveMessageSize());
                    auto auxClient = NYP::NClient::CreateClient(ypOpts);
                    auxClients[clusterName] = auxClient;
                }

                for (const auto& ypClientConfig : configs) {
                   const TString clusterName = ypClientConfig.GetClusterName();

                   NYP::NClient::TClientOptions ypOpts = auxClients[clusterName]->Options();
                   ypOpts
                       .SetSnapshotTimestamp(auxClients[clusterName]->GenerateTimestamp().GetValue(TDuration::Seconds(60)));
                   auto client = NYP::NClient::CreateClient(ypOpts);
                   clients[clusterName] = client;
                   transactionFactories[clusterName] = NYP::NClient::CreateTransactionFactory(*client);
                }

                controller->SetLeadership(true);
                controller->Sync(
                    clients
                    , transactionFactories
                    , MtpQueue_
                    , *AuxMtpQueue_
                    , Logger_.SpawnFrame()
                    , RetryCount_
                    , YpRequestsBatchSize_
                    , MaxSeqFailedObjMngrsCount_
                );
            }
            , MtpQueue_
        );
    }

    for (size_t i = 0; i < controllerFuture.size(); ++i) {
        controllerFuture[i].GetValueSync();
    }

    AuxMtpQueue_->Stop();
    MtpQueue_.Stop();
}

bool TStandaloneController::SafeSync() {
    try {
        Sync();
        return true;
    } catch (...) {
        Cerr << "SafeSync exception messsage: " << CurrentExceptionMessage() << Endl;
        return false;
    }
}

} // namespace NInfra::NController
