#include "error_booster_client.h"

#include <infra/libs/unified_agent_log_backend/unified_agent_log_backend.h>

#include <library/cpp/json/writer/json.h>
#include <library/cpp/protobuf/json/proto2json.h>

#include <contrib/libs/protobuf/src/google/protobuf/reflection.h>

#include <util/string/cast.h>
#include <util/system/env.h>
#include <util/system/hostname.h>

namespace NInfra::NErrorBoosterClient {

TErrorBoosterClient::TErrorBoosterClient(const TErrorBoosterClientConfig& cfg, THolder<IConverter> converter)
    : Level_(FromString<ELogPriority>(cfg.GetLevel()))
    , ProjectName_(cfg.GetProjectName())
    , File_(cfg.GetFile())
    , FillAdditionalWithEventContent_(cfg.GetFillAdditionalWithEventContent())
    , Converter_(std::move(converter))
{
    Backend_ = MakeHolder<NUnifiedAgentLogBackend::TUnifiedAgentLogBackend>(cfg.GetUnifiedAgentLogBackendConfig());
    Logger_ = new TLog(MakeHolder<TThreadedLogBackend>(Backend_.Get(), cfg.GetQueueSize()));
}

bool TErrorBoosterClient::AcceptLevel(ui32 level) const {
    return level <= static_cast<ui8>(Level_);
}

void MessageToMapField(const NProtoBuf::Message& event, google::protobuf::Map<TString, TString>* mapPtr) {
    const NProtoBuf::Descriptor* descriptor = event.GetDescriptor();
    Y_ASSERT(descriptor);

    TVector<const NProtoBuf::FieldDescriptor*> fields;
    event.GetReflection()->ListFields(event, &fields);

    for (auto field : fields) {
        TString name = field->name();
        TString content;
        if (field->is_required() || field->is_optional()) {
            switch (field->cpp_type()) {
                case NProtoBuf::FieldDescriptor::CPPTYPE_INT32: {
                    content = ToString(event.GetReflection()->GetInt32(event, field));
                    break;
                }
                case NProtoBuf::FieldDescriptor::CPPTYPE_INT64: {
                    content = ToString(event.GetReflection()->GetInt64(event, field));
                    break;
                }
                case NProtoBuf::FieldDescriptor::CPPTYPE_UINT32: {
                    content = ToString(event.GetReflection()->GetUInt32(event, field));
                    break;
                }
                case NProtoBuf::FieldDescriptor::CPPTYPE_UINT64: {
                    content = ToString(event.GetReflection()->GetUInt64(event, field));
                    break;
                }
                case NProtoBuf::FieldDescriptor::CPPTYPE_FLOAT: {
                    content = ToString(event.GetReflection()->GetFloat(event, field));
                    break;
                }
                case NProtoBuf::FieldDescriptor::CPPTYPE_DOUBLE: {
                    content = ToString(event.GetReflection()->GetDouble(event, field));
                break;
                }
                case NProtoBuf::FieldDescriptor::CPPTYPE_BOOL: {
                    content = ToString(event.GetReflection()->GetBool(event, field));
                    break;
                }
                case NProtoBuf::FieldDescriptor::CPPTYPE_ENUM: {
                    auto enumValueDescriptor = event.GetReflection()->GetEnum(event, field);
                    content = enumValueDescriptor->name();
                    break;
                }
                case NProtoBuf::FieldDescriptor::CPPTYPE_STRING: {
                    content = event.GetReflection()->GetString(event, field);
                    break;
                }
                default: {}
            }
            if (mapPtr && !content.empty()) {
                (*mapPtr)[name] = content;
            }
        }
    }
}

void TErrorBoosterClient::LogEvent(const NLogEvent::TEventHeader& header, const NProtoBuf::Message& event) {
    if (AcceptLevel(header.GetLogLevel())) {
        TErrorBoosterEvent errorBoosterEvent;
        if (Converter_) {
            if (auto maybeErrorBoosterEvent = Converter_->Convert(header, event); maybeErrorBoosterEvent) {
                errorBoosterEvent = std::move(*maybeErrorBoosterEvent);
                if (!errorBoosterEvent.HasSource()) {
                    errorBoosterEvent.SetSource("generated by user");
                }
            }
        }
        if (!errorBoosterEvent.HasSource()) {
            errorBoosterEvent.SetSource("generated by logger");
        }
        errorBoosterEvent.SetProject(ProjectName_);
        if (errorBoosterEvent.GetFile().empty()) {
            errorBoosterEvent.SetFile(File_);
        }
        errorBoosterEvent.SetTimestamp(header.GetTimestamp() / 1000); // dividing by 1000 to convert microseconds to milliseconds
        errorBoosterEvent.SetLevel(ToString(ELogPriority(header.GetLogLevel())));
        errorBoosterEvent.SetHost(HostName());
        errorBoosterEvent.SetDC(GetEnv("a_dc"));
        if (!errorBoosterEvent.HasMessage()) {
            TString message = event.GetTypeName();
            if (FillAdditionalWithEventContent_) {
                MessageToMapField(event, errorBoosterEvent.MutableAdditional());
            }
            errorBoosterEvent.SetMessage(message);
        }
        LogEvent(errorBoosterEvent);
    }
}

void TErrorBoosterClient::LogEvent(const TErrorBoosterEvent& errorBoosterEvent) {
    TString jsonEvent = NProtobufJson::Proto2Json(errorBoosterEvent, {.UseJsonName = true, .MapAsObject = true});
    Logger_->Write(jsonEvent.data(), jsonEvent.size());
}

void TErrorBoosterClient::SetConverter(THolder<IConverter> converter) {
    Converter_ = std::move(converter);
}

} // namespace NInfra::NErrorBoosterClient
