#include "server.h"

#include <infra/libs/grpc_server/proto/events_decl.ev.pb.h>

#include <util/datetime/base.h>
#include <util/string/cast.h>

namespace NInfra {

TGrpcServer::TGrpcServer(const TGrpcServerConfig& config)
    : Address_(AddressFromConfig(config))
{
    if (config.HasMaxConnectionIdleTime()) {
        ServerBuilder_.AddChannelArgument(GRPC_ARG_MAX_CONNECTION_IDLE_MS, TDuration::Parse(config.GetMaxConnectionIdleTime()).MilliSeconds());
    }

    ServerBuilder_.AddListeningPort(Address_, grpc::InsecureServerCredentials());
}

TGrpcServer::~TGrpcServer() {
    try {
        Stop(nullptr);
    } catch (...) {
    }
}

void TGrpcServer::RegisterService(grpc::Service* service) {
    ServerBuilder_.RegisterService(service);
}

void TGrpcServer::Start(NInfra::TLogFramePtr logFrame) {
    std::unique_ptr<grpc::Server> server = ServerBuilder_.BuildAndStart();
    Server_.Reset(server.release());

    logFrame->LogEvent(NEventlog::TGrpcServerStart(Address_));
}

void TGrpcServer::Stop(NInfra::TLogFramePtr logFrame) {
    if (logFrame) {
        logFrame->LogEvent(NEventlog::TGrpcServerStop());
    }

    if (Server_) {
        Server_->Shutdown();
        Server_->Wait();
    }
}

void TGrpcServer::Wait(NInfra::TLogFramePtr logFrame) {
    with_lock (Mutex_) {
        CondVar_.Wait(Mutex_, [this] { return Shutdown_; });
    }

    Stop(logFrame);
}

void TGrpcServer::Shutdown(NInfra::TLogFramePtr logFrame) {
    logFrame->LogEvent(NEventlog::TGrpcServerShutdownScheduled());

    Shutdown_ = true;
    CondVar_.BroadCast();
}

TString TGrpcServer::AddressFromConfig(const TGrpcServerConfig& config) {
    if (config.HasPort()) {
        return TString::Join("[::]:", ToString(config.GetPort()));
    }
    return "[::]";
}

} // namespace NInfra
