#include "service.h"

#include <infra/libs/service_iface/errors.h>
#include <infra/libs/service_iface/router.h>

#include <library/cpp/cgiparam/cgiparam.h>
#include <library/cpp/http/misc/parsed_request.h>
#include <library/cpp/http/server/response.h>

#include <util/datetime/cputimer.h>
#include <util/network/address.h>
#include <util/string/builder.h>
#include <util/system/hostname.h>

namespace NInfra {

namespace {
class THttpRequestReplier: public TRequestReplier {
public:
    THttpRequestReplier(IRequestRouter& router, IProfileStatsCallbacks* profileCallbacks, IHttpServiceCallbacks* callbacks)
        : Router_(router)
        , ProfileCallbacks_(profileCallbacks)
        , Callbacks_(callbacks)
    {
    }

    bool DoReply(const TReplyParams& params) override {
        TSimpleTimer decodingTimer = TSimpleTimer();
        const TParsedHttpFull parsed(params.Input.FirstLine());
        const TString postData = params.Input.ReadAll();

        if (ProfileCallbacks_) {
            IProfileStatsCallbacks::TProfileStats profileStats;
            profileStats.ProcessingTime = decodingTimer.Get();
            profileStats.IsCompressed = params.Input.ContentEncoded();
            profileStats.DecompressedSize = postData.size();
            profileStats.CompressedSize = profileStats.DecompressedSize;
            params.Input.GetContentLength(profileStats.CompressedSize);

            ProfileCallbacks_->OnRequestBuild(profileStats);
        }

        TVector<std::pair<TString, TString>> attributes(ParsedHeaders);
        attributes.emplace_back(NServiceAttributes::RemoteAddressAttributeName(), GetRemoteAddr());

        THttpResponse response;

        if (!Router_.Has(parsed.Path)) {
            response = THttpResponse(HTTP_NOT_FOUND);
        } else {
            try {
                TRouterResponse rawResponse = Router_.Handle(parsed.Path, postData, attributes);
                response.SetContent(std::move(rawResponse.Content));
                for (const auto& [name, value] : rawResponse.Attributes) {
                    response.AddHeader(THttpInputHeader(name, value));
                }
            } catch (const TNotImplementedError&) {
                response = THttpResponse(HTTP_NOT_IMPLEMENTED);
            } catch (const TAccessDeniedError& ex) {
                response = THttpResponse(HTTP_METHOD_NOT_ALLOWED);
                response.SetContent(CurrentExceptionMessage());
            } catch (const TServiceError& ex) {
                response = THttpResponse(ex.GetHttpCode());
                response.SetContent(CurrentExceptionMessage());
            } catch (...) {
                response = THttpResponse(HTTP_INTERNAL_SERVER_ERROR);
                response.SetContent(CurrentExceptionMessage());
            }
        }

        response.AddHeader("Access-Control-Allow-Origin", "*");

        if (Callbacks_) {
            Callbacks_->OnResponseBuild(response);
        }

        params.Output << response;
        params.Output.Finish();

        return true;
    }

private:
    IRequestRouter& Router_;
    IProfileStatsCallbacks* ProfileCallbacks_;
    IHttpServiceCallbacks* Callbacks_;

    TString GetRemoteAddr() const {
        NAddr::TOpaqueAddr remoteAddr;
        int rc = getpeername(Socket(), remoteAddr.MutableAddr(), remoteAddr.LenPtr());
        return rc == 0 ? PrintHost(remoteAddr) : Default<TString>();
    }
};

} // namespace

TClientRequest* THttpClientFactory::CreateClient() {
    return new THttpRequestReplier(Router_, ProfileStatsCallbacks_.Get(), Callbacks_.Get());
}

void THttpClientFactory::OnFailRequest(int failstate) {
    if (Callbacks_) {
        Callbacks_->OnFailRequest(failstate);
    }
}

void THttpClientFactory::OnException() {
    if (Callbacks_) {
        Callbacks_->OnException();
    }
}

void THttpClientFactory::OnMaxConn() {
    if (Callbacks_) {
        Callbacks_->OnMaxConn();
    }
}

void THttpService::Start(TLogFramePtr logFrame) {
    THttpServer::TOptions opts;
    opts
        .SetPort(Config_.GetPort())
        .SetThreads(Config_.GetThreads())
        .EnableKeepAlive(true)
        .EnableCompression(true)
        .SetClientTimeout(FromString<TDuration>(Config_.GetClientTimeout()));

    if (Config_.HasMaxConnections()) {
        opts.SetMaxConnections(Config_.GetMaxConnections());
    }

    if (Config_.HasMaxQueueSize()) {
        opts.SetMaxQueueSize(Config_.GetMaxQueueSize());
    }

    if (Config_.HasConnectionExpirationTimeout()) {
        opts.ExpirationTimeout = TDuration::Parse(Config_.GetConnectionExpirationTimeout());
    }

    HttpServer_ = MakeHolder<THttpServer>(&ClientFactory_, opts);
    bool res = HttpServer_->Start();
    if (!res) {
        logFrame->LogEvent(ELogPriority::TLOG_WARNING, NLogEvent::TServiceError(
            "http"
            , HttpServer_->GetErrorCode()
            , HttpServer_->GetError()
        ));
        return;
    }

    logFrame->LogEvent(NLogEvent::TStartService("http", ToString(GetHostName()) + ":" + ToString(Config_.GetPort())));
}

void THttpService::Wait(TLogFramePtr logFrame) {
    HttpServer_->Wait();
    logFrame->LogEvent(NLogEvent::TStopService("http"));
}

void THttpService::ShutDown() {
    HttpServer_->Shutdown();
}

void THttpService::Stop() {
    HttpServer_->Stop();
}

size_t THttpService::GetRequestQueueSize() const {
    return HttpServer_->GetRequestQueueSize();
}

size_t THttpService::GetFailQueueSize() const {
    return HttpServer_->GetFailQueueSize();

}
} // namespace NInfra
