#include "service.h"
#include "test_common.h"

#include <library/cpp/http/simple/http_client.h>
#include <library/cpp/testing/common/network.h>
#include <library/cpp/testing/unittest/registar.h>

using namespace NInfra;
using namespace NTestCommon;

namespace {

TString SendRequest(ui16 port, TStringBuf url, TStringBuf body, bool post) {
    TStringStream outputStream;
    TSimpleHttpClient client("http://localhost", port);

    if (!post) {
        client.DoGet(url, &outputStream);
    } else {
        client.DoPost(url, body, &outputStream);
    }

    return outputStream.Str();
}

TString SendGetRequest(ui16 port, TStringBuf url) {
    return SendRequest(port, url, "", /* post = */ false);
}

TString SendPostRequest(ui64 port, TStringBuf url, TStringBuf body) {
    return SendRequest(port, url, body, /* post = */ true);
}

class TFakeRouter: public IRequestRouter {
public:
    TFakeRouter() {
        for (TStringBuf path : {"/hello", "/world"}) {
            Counter_[path] = 0;
        }
    }

    bool Has(TStringBuf path) const override {
        return Counter_.contains(path);
    }

    TRouterResponse Handle(TStringBuf path, const TString& postData, const TVector<std::pair<TString, TString>>&) const override {
        TStringBuilder builder;
        builder << "ACK " << path << " " << Counter_[path];

        if (!postData.empty()) {
            builder << " " << postData;
        }

        ++Counter_[path];

        return {builder, {}};
    }

private:
    mutable THashMap<TString, size_t> Counter_;
};

} // namespace

Y_UNIT_TEST_SUITE(HttpService) {
    Y_UNIT_TEST(TestAdminActions) {
        const NTesting::TPortHolder port = NTesting::GetFreePort();
        const THttpServiceConfig config = GenerateHttpServiceConfig(port);

        TLoggerConfig loggerConfig;
        TLogger logger(loggerConfig);
        auto fakeFrame = logger.SpawnFrame();

        THttpService httpService(config, MakeSimpleShared<TFakeRouter>());
        httpService.Start(fakeFrame);

        for (TStringBuf path : {"/hello", "/world"}) {
            const TString s = TStringBuilder() << "ACK " << path << " 0";
            UNIT_ASSERT_STRINGS_EQUAL(s, SendGetRequest(config.GetPort(), path));
        }

        for (TStringBuf path : {"/hello", "/world"}) {
            const TString s = TStringBuilder() << "ACK " << path << " 1";
            UNIT_ASSERT_STRINGS_EQUAL(s, SendGetRequest(config.GetPort(), path));
        }

        for (TStringBuf path : {"/hello", "/world"}) {
            const TString s = TStringBuilder() << "ACK " << path << " 2 POST";
            UNIT_ASSERT_STRINGS_EQUAL(s, SendPostRequest(config.GetPort(), path, "POST"));
        }

        for (TStringBuf path : {"/no", "/path"}) {
            UNIT_ASSERT_EXCEPTION(SendGetRequest(config.GetPort(), path), THttpRequestException);
        }

        httpService.ShutDown();
    }
}
