package org.apache.logging.log4j.core.appender;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;

import ch.qos.logback.classic.Level;
import ch.qos.logback.classic.LoggerContext;
import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.classic.spi.IThrowableProxy;
import ch.qos.logback.classic.spi.LoggerContextVO;
import ch.qos.logback.classic.spi.ThrowableProxy;
import ch.qos.logback.core.Context;
import org.apache.logging.log4j.core.Appender;
import org.apache.logging.log4j.core.Core;
import org.apache.logging.log4j.core.Filter;
import org.apache.logging.log4j.core.Layout;
import org.apache.logging.log4j.core.LogEvent;
import org.apache.logging.log4j.core.config.plugins.Plugin;
import org.apache.logging.log4j.core.config.plugins.PluginAttribute;
import org.apache.logging.log4j.core.config.plugins.PluginElement;
import org.apache.logging.log4j.core.config.plugins.PluginFactory;
import org.apache.logging.log4j.core.layout.PatternLayout;
import org.slf4j.Marker;

import ru.yandex.misc.net.HostnameUtils;
import ru.yandex.qe.logging.remote.qloud.QloudRemoteAppender;

@Plugin(name = "AsyncTcpSocketAppender",
        category = Core.CATEGORY_NAME,
        elementType = Appender.ELEMENT_TYPE, printObject = true)
public class AsyncTcpSocketAppender extends AbstractAppender {

    private final LocalQloudRemoteAppender appender;

    @PluginFactory
    public static AsyncTcpSocketAppender createAppender(@PluginAttribute("name") String name,
                                                        @PluginAttribute("ignoreExceptions") boolean ignoreExceptions,
                                                        @PluginAttribute("host") String host,
                                                        @PluginAttribute("port") String port,
                                                        @PluginElement("Layout") Layout layout,
                                                        @PluginElement("Filters") Filter filter) {

        if (name == null) {
            LOGGER.error("No name provided for AsyncTcpSocketAppender");
            return null;
        }

        if (layout == null) {
            layout = PatternLayout.createDefaultLayout();
        }

        Map<String, String> env = new HashMap<>(System.getenv());
        env.put("QLOUD_LOGGER_PORT", port);
        env.put("QLOUD_LOGGER_HOST", host);
        env.put("QLOUD_HOSTNAME", HostnameUtils.localHostname());

        LocalQloudRemoteAppender appender = new LocalQloudRemoteAppender(new LoggerContext(), env);
        appender.start();
        return new AsyncTcpSocketAppender(name, filter, layout, ignoreExceptions, appender);
    }

    private AsyncTcpSocketAppender(String name, Filter filter, Layout<? extends Serializable> layout, boolean ignoreExceptions, LocalQloudRemoteAppender appender) {
        super(name, filter, layout, ignoreExceptions);
        this.appender = appender;
    }

    @Override
    public void append(LogEvent event) {
        appender.append(new LogEventWrapper(event));
    }

    private static class LocalQloudRemoteAppender extends QloudRemoteAppender {

        public LocalQloudRemoteAppender(Context context, Map<String, String> env) {
            super(context, env);
        }

        @Override
        public void append(ILoggingEvent origEvent) {
            super.append(origEvent);
        }
    }

    private static class LogEventWrapper implements ILoggingEvent {

        private final LogEvent event;

        private LogEventWrapper(LogEvent event) {
            this.event = event;
        }

        @Override
        public String getThreadName() {
            return event.getThreadName();
        }

        @Override
        public Level getLevel() {
            if (event.getLevel() != null) {
                org.apache.logging.log4j.Level level = event.getLevel();
                if (org.apache.logging.log4j.Level.FATAL.equals(level) || org.apache.logging.log4j.Level.ERROR.equals(level)) {
                    return Level.ERROR;
                } else if (org.apache.logging.log4j.Level.WARN.equals(level)) {
                    return Level.WARN;
                } else if (org.apache.logging.log4j.Level.INFO.equals(level)) {
                    return Level.INFO;
                } else if (org.apache.logging.log4j.Level.OFF.equals(level)) {
                    return Level.OFF;
                } else if (org.apache.logging.log4j.Level.ALL.equals(level)) {
                    return Level.ALL;
                } else {
                    return Level.DEBUG;
                }
            }
            return Level.DEBUG;
        }

        @Override
        public String getMessage() {
            if (event.getMessage() == null)
                return null;
            return event.getMessage().getFormattedMessage();
        }

        @Override
        public Object[] getArgumentArray() {
            return new Object[0];
        }

        @Override
        public String getFormattedMessage() {
            if (event.getMessage() == null)
                return null;
            return event.getMessage().getFormattedMessage();
        }

        @Override
        public String getLoggerName() {
            return event.getLoggerName();
        }

        @Override
        public LoggerContextVO getLoggerContextVO() {
            return new LoggerContextVO(event.getLoggerName(), event.getContextData().toMap(), event.getTimeMillis());
        }

        @Override
        public IThrowableProxy getThrowableProxy() {
            if (event.getThrownProxy() == null)
                return null;
            return new ThrowableProxy(event.getThrownProxy().getThrowable());
        }

        @Override
        public StackTraceElement[] getCallerData() {
            if (event.getThrownProxy() == null)
                return null;
            return event.getThrownProxy().getStackTrace();
        }

        @Override
        public boolean hasCallerData() {
            if (event.getThrownProxy() == null)
                return false;
            return event.getThrownProxy().getStackTrace() != null;
        }

        @Override
        public Marker getMarker() {
            return null;
        }

        @Override
        public Map<String, String> getMDCPropertyMap() {
            return null;
        }

        @Override
        public Map<String, String> getMdc() {
            return null;
        }

        @Override
        public long getTimeStamp() {
            return event.getTimeMillis();
        }

        @Override
        public void prepareForDeferredProcessing() {
        }
    }
}
