package ru.yandex.infra.controller.concurrent;

import java.time.Duration;
import java.util.Optional;
import java.util.OptionalLong;

import com.google.common.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.locks.CypressLockProvider;
import ru.yandex.misc.io.http.Timeout;

public class CypressLockingService implements LockingService {
    private static final Logger LOG = LoggerFactory.getLogger(CypressLockingService.class);
    private final CypressLockProvider.LockImpl lock;
    private final YPath lockPath;
    private final YPath epochPath;
    private final Runnable lostLockCallback;

    public CypressLockingService(String proxy, String token, YPath lockPath, YPath epochPath) {
        this(proxy, token, lockPath, epochPath, Duration.ofSeconds(10), Duration.ofSeconds(1),
                new Timeout(1000, 500), 50, null);
    }

    // Lock-holding YT transaction is created with timeout 2*pingInterval; it is pinged each pingInterval by default
    // or each retryPingInterval if previous ping was unsuccessful.
    public CypressLockingService(String proxy, String token, YPath lockPath, YPath epochPath,
                                 Duration pingInterval, Duration retryPingInterval, Timeout timeout,
                                 int maxConnectionCount, Runnable lostLockCallback) {
        if (pingInterval.compareTo(retryPingInterval.multipliedBy(2)) < 0) {
            String message = String.format("Retry interval %s is too small for ping interval %s, should be at least 2" +
                            " times longer",
                    retryPingInterval, pingInterval);
            throw new IllegalArgumentException(message);
        }
        CypressLockProvider provider = new CypressLockProvider(proxy, token, Optional.empty(), lockPath.parent(),
                pingInterval, retryPingInterval, timeout, maxConnectionCount);

        this.lockPath = lockPath;
        this.epochPath = epochPath;
        this.lostLockCallback = lostLockCallback;
        lock = provider.createLock(lockPath.name(), this::onPingerException);
    }

    private void onPingerException(Exception exception) {
        LOG.error("Exception during pinging YT lock transaction", exception);
        if (!lock.isTaken()) {
            LOG.warn("Leadership has been lost: {}", this);
            if (lostLockCallback != null) {
                lostLockCallback.run();
            }
        }
    }

    @Override
    public long lock() {
        LOG.info("Will try to acquire {}", toString());
        lock.lock();
        lock.initEpoch(epochPath);
        OptionalLong currentEpoch = OptionalLong.empty();
        while (currentEpoch.isEmpty()) {
            try {
                currentEpoch = OptionalLong.of(lock.tryIncrementEpoch(epochPath));
            } catch (Exception e) {
                if (lock.isTaken()) {
                    LOG.warn("Cannot increment epoch, but lock still alive", e);
                } else {
                    lock.lock();
                }
            }
        }
        LOG.info("{} acquired", toString());
        return currentEpoch.getAsLong();
    }

    @Override
    public boolean isLocked() {
        return lock.isTaken();
    }

    @Override
    public GUID getTransactionId() {
        return lock.getTransactionId();
    }

    @Override
    public String toString() {
        return "Leadership YT lock at " + this.lockPath;
    }

    @VisibleForTesting
    public CypressLockProvider.LockImpl getInternalLock() {
        return lock;
    }
}
