package ru.yandex.infra.controller.concurrent;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import com.codahale.metrics.MetricRegistry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.controller.metrics.MetricUtils;
import ru.yandex.qe.telemetry.metrics.Gauges;

public class LeaderServiceImpl implements LeaderService {
    private static final Logger LOG = LoggerFactory.getLogger(LeaderServiceImpl.class);
    private final LockingService lockingService;
    private final List<Runnable> leadershipAcquiredCallbacks = new ArrayList<>();
    private final List<Runnable> leadershipLostCallbacks = new ArrayList<>();
    private final List<Runnable> processingAllowedCallbacks = new ArrayList<>();
    private volatile boolean isProcessingAllowed = false;
    private volatile boolean isLeader = false;
    private volatile Long currentEpoch = null;

    public LeaderServiceImpl(String serviceName, LockingService lockingService, MetricRegistry metricRegistry) {
        this.lockingService = lockingService;
        Gauges.forSupplier(
                metricRegistry,
                serviceName + METRIC_DELIMITER + LEADER_LOCK_METRIC,
                Set.of("ammx"),
                () -> MetricUtils.booleanToInt(lockingService.isLocked())
        );
        Gauges.forSupplier(
                metricRegistry,
                serviceName + METRIC_DELIMITER + PROCESSING_ALLOWED_METRIC,
                () -> MetricUtils.booleanToInt(isProcessingAllowed)
        );
        Gauges.forSupplier(
                metricRegistry,
                serviceName + METRIC_DELIMITER + EPOCH_METRIC,
                this::getCurrentEpoch
        );
    }

    @Override
    public void ensureLeadership() {
        if (!lockingService.isLocked()) {
            LOG.info("ensureLeadership: currently without lock at {}", lockingService);
            synchronized (this) {
                if (isLeader) {
                    leadershipLostCallbacks.forEach(Runnable::run);
                    isLeader = false;
                }
            }
            if (isProcessingAllowed) {
                isProcessingAllowed = false;
                LOG.warn("Lock {} lost, processing is forbidden until explicitly allowed again", lockingService);
            }

            currentEpoch = lockingService.lock();
            isLeader = true;
            LOG.info("Lock {} acquired", lockingService);
            synchronized (this) {
                leadershipAcquiredCallbacks.forEach(Runnable::run);
            }
        }
    }

    @Override
    public synchronized void allowProcessing() {
        if (!isProcessingAllowed) {
            LOG.info("Processing has been forbidden before and is allowed now");
            isProcessingAllowed = true;
            processingAllowedCallbacks.forEach(Runnable::run);
        }
    }

    @Override
    public boolean isProcessingAllowed() {
        return isProcessingAllowed;
    }

    @Override
    public boolean isLeader() {
        return lockingService.isLocked();
    }

    @Override
    public synchronized void addProcessingAllowedCallback(Runnable callback) {
        processingAllowedCallbacks.add(callback);
    }

    @Override
    public synchronized void addLeadershipAcquiredCallback(Runnable callback) {
        leadershipAcquiredCallbacks.add(callback);
    }

    @Override
    public void addLeadershipLostCallback(Runnable callback) {
        leadershipLostCallbacks.add(callback);
    }

    @Override
    public Long getCurrentEpoch() {
        return currentEpoch;
    }

}
