package ru.yandex.infra.controller.metrics;

import java.util.HashMap;
import java.util.Map;

import com.codahale.metrics.Metric;
import com.codahale.metrics.MetricRegistry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.controller.concurrent.LeaderService;

/**
 * Reports metrics to underlying MetricRegistry only if leadership is currently acquired.
 * Metrics are removed on leadership loss and restored on reacquisition
 */
public class LeaderGaugeRegistry implements GaugeRegistry {
    private static final Logger LOG = LoggerFactory.getLogger(LeaderGaugeRegistry.class);

    private final Map<String, Metric> managedGauges = new HashMap<>();
    private final MetricRegistry metricRegistry;
    private final LeaderService leaderService;

    public LeaderGaugeRegistry(MetricRegistry metricRegistry, LeaderService leaderService) {
        this.metricRegistry = metricRegistry;
        this.leaderService = leaderService;
        leaderService.addLeadershipAcquiredCallback(this::onLeadershipAcquired);
    }

    @Override
    public synchronized Metric register(String name, Metric metric) throws IllegalArgumentException {

        Metric previousMetric = managedGauges.get(name);
        if (previousMetric != null) {
            LOG.warn("Trying to add second metric with the same name: {}", name);
            return previousMetric;
        }

        managedGauges.put(name, metric);
        if (leaderService.isLeader()) {
            metricRegistry.register(name, metric);
        }
        return metric;
    }

    @Override
    public synchronized boolean remove(String name) {
        metricRegistry.remove(name);
        return managedGauges.remove(name) != null;
    }

    private synchronized void onLeadershipAcquired() {
        managedGauges.forEach(metricRegistry::register);
    }
}
