package ru.yandex.infra.controller.metrics;

import java.util.Collection;
import java.util.Collections;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.typesafe.config.Config;

public interface MetricConfig {

    String ENABLED_FIELD = "enabled";
    String AGGREGATIONS_FIELD = "aggregations";
    String PREFIX_FIELD = "prefix";

    String PREFIX_DELIMITER = ".";

    MetricConfig ENABLED_EMPTY = new Empty(true), DISABLED_EMPTY = new Empty(false);

    MetricConfig getConfig(String name);

    boolean isEnabled();

    String getNamePrefix();
    Collection<String> getAggregations();

    final class Empty implements MetricConfig {
        private final boolean enabled;

        private Empty(boolean enabled) {
            this.enabled = enabled;
        }

        @Override
        public MetricConfig getConfig(String name) {
            return DISABLED_EMPTY;
        }

        @Override
        public boolean isEnabled() {
            return enabled;
        }

        @Override
        public String getNamePrefix() {
            return "";
        }

        @Override
        public Collection<String> getAggregations() {
            return Collections.emptyList();
        }
    }

    class Impl implements MetricConfig {

        final Config config;

        final boolean enabled;
        final String namePrefix;
        final Collection<String> aggregations;

        public Impl(Config config) {
            this(config, ENABLED_EMPTY);
        }

        public Impl(Config config, MetricConfig parentConfig) {
            this.config = config;

            boolean configEnabled = !config.hasPath(ENABLED_FIELD) || config.getBoolean(ENABLED_FIELD);
            this.enabled = parentConfig.isEnabled() && configEnabled;

            StringBuilder namePrefixBuilder = new StringBuilder();
            namePrefixBuilder.append(parentConfig.getNamePrefix());
            if (namePrefixBuilder.length() > 0) namePrefixBuilder.append(PREFIX_DELIMITER);

            String configNamePrefix = config.hasPath(PREFIX_FIELD) ? config.getString(PREFIX_FIELD) : "";
            namePrefixBuilder.append(configNamePrefix);

            this.namePrefix = namePrefixBuilder.toString();

            Collection<String> parentAggregations = parentConfig.getAggregations();
            Collection<String> configAggregations = config.hasPath(AGGREGATIONS_FIELD) ? config.getStringList(AGGREGATIONS_FIELD) : Collections.emptyList();

            this.aggregations = Stream.of(parentAggregations, configAggregations).flatMap(Collection::stream).collect(Collectors.toList());
        }

        @Override
        public MetricConfig getConfig(String name) {
            if (!config.hasPath(name)) {
                return DISABLED_EMPTY;
            }

            Config childConfig = config.getConfig(name);
            return new Impl(childConfig, this);
        }

        @Override
        public boolean isEnabled() {
            return enabled;
        }

        @Override
        public String getNamePrefix() {
            return namePrefix;
        }

        @Override
        public Collection<String> getAggregations() {
            return aggregations;
        }
    }
}
