package ru.yandex.infra.controller.metrics;

import java.util.Collection;

import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.jvm.FileDescriptorRatioGauge;
import com.codahale.metrics.jvm.GarbageCollectorMetricSet;
import com.codahale.metrics.jvm.MemoryUsageGaugeSet;
import com.codahale.metrics.jvm.ThreadStatesGaugeSet;
import com.typesafe.config.Config;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.core.Appender;
import org.apache.logging.log4j.core.LoggerContext;
import org.apache.logging.log4j.core.config.Configuration;

import ru.yandex.qe.telemetry.memory.GCMetricsSet;
import ru.yandex.qe.telemetry.metrics.yasm.YasmMetricRegistry;

public class MetricUtils {
    public static int booleanToInt(boolean value) {
        return value ? 1 : 0;
    }

    public static final MetricConfig DEFAULT_METRIC_CONFIG = MetricConfig.ENABLED_EMPTY;

    public static MetricRegistry buildMetricRegistry() {
        return buildMetricRegistry(DEFAULT_METRIC_CONFIG);
    }

    public static MetricRegistry buildMetricRegistry(Config metricsConfig) {
        return buildMetricRegistry(new MetricConfig.Impl(metricsConfig));
    }

    public static MetricRegistry buildMetricRegistry(MetricConfig metricConfig) {
        MetricRegistry metricRegistry = new YasmMetricRegistry();

        if (metricConfig.isEnabled()) {
            metricRegistry.register("jvm.gc.yasm", new GCMetricsSet());
            metricRegistry.register("jvm.gc", new GarbageCollectorMetricSet());
            metricRegistry.register("jvm.memory", new MemoryUsageGaugeSet());
            metricRegistry.register("jvm.thread-states", new ThreadStatesGaugeSet());
            metricRegistry.register("jvm.fd.usage", new FileDescriptorRatioGauge());

            MetricConfig logMetricConfig = metricConfig.getConfig("log");
            registerLogMetric(metricRegistry, logMetricConfig);
        }

        return metricRegistry;
    }

    public static void registerLogMetric(MetricRegistry metricRegistry, MetricConfig logMetricConfig) {
        if (!logMetricConfig.isEnabled()) return;

        String metricNamePrefix = logMetricConfig.getNamePrefix();
        Collection<String> aggregations = logMetricConfig.getAggregations();

        registerLog4j2Metric(metricRegistry, metricNamePrefix, aggregations);
    }

    private static void registerLog4j2Metric(MetricRegistry metricRegistry, String metricNamePrefix, Collection<String> aggregations) {
        Appender appender = new Log4j2MetricAppender(metricRegistry, metricNamePrefix, aggregations);
        appender.start();

        LoggerContext context = (LoggerContext) LogManager.getContext(false);
        Configuration config = context.getConfiguration();

        // filter can be null according to https://metrics.dropwizard.io/4.1.2/manual/log4j.html
        config.getLoggerConfig(LogManager.ROOT_LOGGER_NAME).addAppender(appender, Level.ALL, null);
        context.updateLoggers(config);
    }
}
