package ru.yandex.infra.controller.servlets;

import java.io.IOException;
import java.lang.reflect.Field;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.codahale.metrics.MetricRegistry;
import com.google.common.base.Stopwatch;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.qe.telemetry.metrics.Gauges;
import ru.yandex.qe.telemetry.metrics.yasm.YasmReporter;
import ru.yandex.qe.telemetry.metrics.yasm.YasmUtils;

public class UnistatServlet extends HttpServlet {
    private static final Logger LOG = LoggerFactory.getLogger(UnistatServlet.class);

    static final String METRIC_UNISTAT_COUNT = "unistat_count";
    static final String METRIC_UNISTAT_EXECUTION_TIME = "unistat_execution_time_ms";
    static final String METRIC_METRICS_COUNT = "metrics_count";

    private final AtomicLong metricUnistatCount = new AtomicLong();
    private final AtomicLong metricUnistatExecutionTimeNanos = new AtomicLong();

    private final YasmReporter reporter;

    public UnistatServlet(MetricRegistry registry) {
        reporter = new YasmReporter();
        try {
            // Need this because the class is too coupled to spring DI.
            Field metricRegistryField = YasmReporter.class.getDeclaredField("metricRegistry");
            metricRegistryField.setAccessible(true);
            metricRegistryField.set(reporter, registry);

            Gauges.forSupplier(registry, METRIC_UNISTAT_COUNT, Set.of("dmmm"), metricUnistatCount::get);
            Gauges.forSupplier(registry, METRIC_UNISTAT_EXECUTION_TIME, Set.of("dmmm"), () -> metricUnistatExecutionTimeNanos.get() / 1000000L);
            Gauges.forSupplier(registry, METRIC_METRICS_COUNT, Set.of("axxx"), () -> registry.getMetrics().size());


        } catch (NoSuchFieldException | IllegalAccessException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws IOException {
        Stopwatch sw = Stopwatch.createStarted();
        try {
            metricUnistatCount.incrementAndGet();
            LOG.debug("/unistat request from {}", req.getRemoteHost());
            YasmUtils.processUnitstatRequest(req, resp, reporter);
        } finally {
            metricUnistatExecutionTimeNanos.addAndGet(sw.elapsed(TimeUnit.NANOSECONDS));
        }
    }
}
