package ru.yandex.infra.controller.util;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;

import com.codahale.metrics.MetricRegistry;
import com.typesafe.config.Config;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.controller.YtSettings;
import ru.yandex.infra.controller.concurrent.CypressLockingService;
import ru.yandex.infra.controller.concurrent.DummyLockingService;
import ru.yandex.infra.controller.concurrent.LeaderService;
import ru.yandex.infra.controller.concurrent.LeaderServiceImpl;
import ru.yandex.infra.controller.concurrent.LockingService;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.yt.ytclient.rpc.RpcOptions;

public class ConfigUtils {
    private static final Logger LOG = LoggerFactory.getLogger(ConfigUtils.class);

    public static LeaderService leaderService(String serviceName, Config config, MetricRegistry metricRegistry, YtSettings ytSettings) {
        LockingService lockingService = config.getBoolean("use_lock") ?
                new CypressLockingService(ytSettings.getProxy(), ytSettings.getToken(),
                        YPath.simple(config.getString("lock_path")),
                        YPath.simple(config.getString("epoch_path")),
                        config.getDuration("ping_interval"),
                        config.getDuration("retry_ping_interval"),
                        new Timeout(config.getDuration("so_timeout").toMillis(),
                                config.getDuration("connection_timeout").toMillis()),
                        config.getInt("max_connection_count"),
                        ConfigUtils::onLostLeadership
                ) : new DummyLockingService(true);

        return new LeaderServiceImpl(serviceName, lockingService, metricRegistry);
    }

    public static String token(String path) {
        try (FileInputStream fis = new FileInputStream(path);
             BufferedReader reader = new BufferedReader(new InputStreamReader(fis))) {
            return reader.readLine();
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private static void onLostLeadership() {
        LOG.error("Leadership lock lost, performing forced shutdown");
        ExitUtils.gracefulExit(ExitUtils.LEADERSHIP_LOST);
    }

    public static YtSettings ytSettings(Config ytConfig) {
        return new YtSettings(
                ytConfig.getString("yt_proxy"),
                token(ytConfig.getString("token_file")),
                ytConfig.getString("user"),
                rpcOptions(ytConfig.getConfig("rpc_options")));
    }

    public static RpcOptions rpcOptions(Config rpcConfig) {
        RpcOptions rpcOptions = new RpcOptions();

        if (rpcConfig.hasPath("global_timeout")) {
            rpcOptions.setGlobalTimeout(rpcConfig.getDuration("global_timeout"));
        }

        if (rpcConfig.hasPath("streaming_read_timeout")) {
            rpcOptions.setStreamingReadTimeout(rpcConfig.getDuration("streaming_read_timeout"));
        }

        if (rpcConfig.hasPath("streaming_write_timeout")) {
            rpcOptions.setStreamingWriteTimeout(rpcConfig.getDuration("streaming_write_timeout"));
        }

        if (rpcConfig.hasPath("acknowledgement_timeout")) {
            rpcOptions.setAcknowledgementTimeout(rpcConfig.getDuration("acknowledgement_timeout"));
        }

        if (rpcConfig.hasPath("ping_timeout")) {
            rpcOptions.setPingTimeout(rpcConfig.getDuration("ping_timeout"));
        }

        if (rpcConfig.hasPath("failover_timeout")) {
            rpcOptions.setFailoverTimeout(rpcConfig.getDuration("failover_timeout"));
        }

        if (rpcConfig.hasPath("proxy_update_timeout")) {
            rpcOptions.setProxyUpdateTimeout(rpcConfig.getDuration("proxy_update_timeout"));
        }

        if (rpcConfig.hasPath("rpc_client_selection_timeout")) {
            rpcOptions.setRpcClientSelectionTimeout(rpcConfig.getDuration("rpc_client_selection_timeout"));
        }

        if (rpcConfig.hasPath("streaming_window_size")) {
            rpcOptions.setStreamingWindowSize(rpcConfig.getBytes("streaming_window_size").intValue());
        }

        return rpcOptions;
    }
}
