package ru.yandex.infra.controller.util;

import java.time.Duration;
import java.util.concurrent.TimeUnit;

import com.codahale.metrics.MetricRegistry;
import com.typesafe.config.Config;

import ru.yandex.infra.controller.metrics.GaugeRegistry;
import ru.yandex.infra.controller.metrics.GolovanableGauge;
import ru.yandex.infra.controller.yp.ReadonlyYpRawObjectService;
import ru.yandex.infra.controller.yp.Selector;
import ru.yandex.yp.YpInstance;
import ru.yandex.yp.YpRawClient;
import ru.yandex.yp.YpRawClientBuilder;
import ru.yandex.yp.YpRawObjectService;

import static ru.yandex.infra.controller.util.ConfigUtils.token;

public class YpUtils {
    private static final String METRIC_REQUESTS_COUNT = "requests_count";
    private static final String METRIC_FAILED_REQUESTS_COUNT = "failed_requests_count";
    private static final String METRIC_ALIVE_MASTERS_COUNT = "alive_masters_count";

    public static YpRawObjectService ypRawClient(Config instanceConfig,
                                                 Config ypConfig,
                                                 GaugeRegistry gaugeRegistry,
                                                 boolean isReadonlyMode) {
        String token = token(ypConfig.getString("token_file"));
        Duration mastersTimeout = ypConfig.getDuration("masters_request_timeout");
        Duration mastersDiscoveryPeriod = ypConfig.getDuration("masters_discovery_period");
        Duration requestTimeout = ypConfig.getDuration("request_timeout");
        String host = instanceConfig.getString("host");
        int port = instanceConfig.getInt("port");

        YpInstance instance = new YpInstance(host, port);
        YpRawClient rawClient = new YpRawClientBuilder(instance, () -> token)
                .setTimeout(requestTimeout.toNanos(), TimeUnit.NANOSECONDS)
                .setMastersRequestTimeout(mastersTimeout.toNanos(), TimeUnit.NANOSECONDS)
                .setUseMasterDiscovery(ypConfig.getBoolean("use_master_discovery"))
                .setMastersAutoDiscoveryPeriod(mastersDiscoveryPeriod.toNanos(), TimeUnit.NANOSECONDS)
                .setUsePlaintext(!ypConfig.getBoolean("secure"))
                .setMaxInboundMessageSize(ypConfig.getBytes("inbound_message_size_limit").intValue())
                .setMaxInboundMetadataSize(ypConfig.getBytes("inbound_metadata_size_limit").intValue())
                .build();

        String metricsPrefix = "yp." + host.split("\\.")[0];
        gaugeRegistry.register(MetricRegistry.name(metricsPrefix, METRIC_REQUESTS_COUNT),
                new GolovanableGauge<>(() -> rawClient.getStatistics().getDiscoveryRequestsCount(), "dmmm"));
        gaugeRegistry.register(MetricRegistry.name(metricsPrefix, METRIC_FAILED_REQUESTS_COUNT),
                new GolovanableGauge<>(() -> rawClient.getStatistics().getFailedDiscoveryRequestsCount(), "dmmm"));
        gaugeRegistry.register(MetricRegistry.name(metricsPrefix, METRIC_ALIVE_MASTERS_COUNT),
                new GolovanableGauge<>(() -> rawClient.getStatistics().getAliveMastersCount().orElse(-1), "annv"));

        YpRawObjectService ypRawObjectService = rawClient.objectService();
        return isReadonlyMode ? new ReadonlyYpRawObjectService(ypRawObjectService) : ypRawObjectService;
    }

    public static class CommonSelectors {
        public static final Selector SPEC_STATUS_META = new Selector.Builder()
                .withMeta()
                .withSpec()
                .withStatus()
                .withTimestamps()
                .build();

        public static final Selector SPEC_STATUS_META_LABELS = new Selector.Builder()
                .withMeta()
                .withSpec()
                .withStatus()
                .withLabels()
                .withTimestamps()
                .build();

        public static final Selector SPEC_META = new Selector.Builder()
                .withSpec()
                .withMeta()
                .build();

        public static final Selector SPEC_LABELS = new Selector.Builder()
                .withSpec()
                .withLabels()
                .withTimestamps()
                .build();

        public static final Selector SPEC_WITH_TIMESTAMP = new Selector.Builder()
                .withSpec()
                .withTimestamps()
                .build();

        public static final Selector SPEC = new Selector.Builder()
                .withSpec()
                .build();

        public static final Selector META = new Selector.Builder()
                .withMeta()
                .build();
    }
}
