package ru.yandex.infra.controller.yp;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import com.google.common.base.MoreObjects;
import com.google.protobuf.Message;

import ru.yandex.infra.controller.dto.Acl;


public class CreateObjectRequest<Spec extends Message> {
    private final Spec spec;
    private final Optional<Acl> acl;
    private final Map<String, Object> labels;
    private final Map<String, String> specificMetaFields;

    public CreateObjectRequest(Spec spec, Optional<Acl> acl, Map<String, Object> labels,
            Map<String, String> specificMetaFields) {
        this.spec = spec;
        this.acl = acl;
        this.labels = labels;
        this.specificMetaFields = specificMetaFields;
    }

    public Spec getSpec() {
        return spec;
    }

    public Optional<Acl> getAcl() {
        return acl;
    }

    public Map<String, Object> getLabels() {
        return labels;
    }

    public Map<String, String> getSpecificMetaFields() {
        return specificMetaFields;
    }

    public static class Builder<Spec extends Message> {
        private final Spec spec;
        private Optional<Acl> acl = Optional.empty();
        private Map<String, Object> labels = new HashMap<>();
        private Map<String, String> specificMetaFields = new HashMap<>();

        public Builder(Spec spec) {
            this.spec = spec;
        }

        public Optional<Acl> getAcl() {
            return acl;
        }

        public Builder<Spec> setAcl(Acl acl) {
            this.acl = Optional.of(acl);
            return this;
        }

        public Builder<Spec> setLabels(Map<String, Object> labels) {
            this.labels = new HashMap<>(labels);
            return this;
        }

        public Builder<Spec> addLabel(String key, Object value) {
            labels.put(key, value);
            return this;
        }

        public Builder<Spec> setSpecificMetaFields(Map<String, String> specificMetaFields) {
            this.specificMetaFields = new HashMap<>(specificMetaFields);
            return this;
        }

        public Builder<Spec> addSpecificMetaField(String key, String value) {
            specificMetaFields.put(key, value);
            return this;
        }

        public CreateObjectRequest<Spec> build() {
            return new CreateObjectRequest<>(spec, acl, labels, specificMetaFields);
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof CreateObjectRequest)) {
            return false;
        }
        CreateObjectRequest<?> that = (CreateObjectRequest<?>) o;
        return Objects.equals(spec, that.spec) &&
                Objects.equals(acl, that.acl) &&
                Objects.equals(labels, that.labels) &&
                Objects.equals(specificMetaFields, that.specificMetaFields);
    }

    @Override
    public int hashCode() {
        return Objects.hash(spec, acl, labels, specificMetaFields);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("spec", spec)
                .add("acl", acl)
                .add("labels", labels)
                .add("specificMetaFields", specificMetaFields)
                .toString();
    }
}
