package ru.yandex.infra.controller.yp;

import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.protobuf.Message;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.controller.dto.SchemaMeta;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.yp.YpRawObjectService;
import ru.yandex.yp.model.YpGetStatement;
import ru.yandex.yp.model.YpObjectType;
import ru.yandex.yp.model.YpTransaction;
import ru.yandex.yp.model.YpTypedId;

import static java.util.concurrent.CompletableFuture.completedFuture;

public class EpochDecoratorRepository<Meta extends SchemaMeta, Spec extends Message, Status extends Message>
        implements YpObjectTransactionalRepository<Meta, Spec, Status> {
    private static final Logger LOG = LoggerFactory.getLogger(EpochDecoratorRepository.class);

    private final YpObjectTransactionalRepository<Meta, Spec, Status> ypRepository;
    private final YpObjectType objectType;
    private final Supplier<Long> epochGetter;
    private final String epochKey;
    private final String epochPath;

    public EpochDecoratorRepository(YpObjectTransactionalRepository<Meta, Spec, Status> ypRepository,
            YpObjectType objectType,
            Supplier<Long> epochGetter,
            String epochKey) {
        this.ypRepository = ypRepository;
        this.objectType = objectType;
        this.epochGetter = epochGetter;
        this.epochKey = epochKey;
        this.epochPath = getEpochPath(epochKey);
    }

    private CreateObjectRequest<Spec> patchCreateRequest(CreateObjectRequest<Spec> request) {
        CreateObjectRequest.Builder<Spec> patchedRequest = new CreateObjectRequest.Builder<>(request.getSpec())
                .setLabels(request.getLabels())
                .setSpecificMetaFields(request.getSpecificMetaFields())
                .addLabel(epochKey, epochGetter.get());
        request.getAcl().ifPresent(patchedRequest::setAcl);
        return patchedRequest.build();
    }

    @Override
    public YpTransactionClient getTransactionClient() {
        return ypRepository.getTransactionClient();
    }

    @Override
    public YpRawObjectService getRawClient() {
        return ypRepository.getRawClient();
    }

    @Override
    public YpObjectType getObjectType() {
        return ypRepository.getObjectType();
    }

    @Override
    public CompletableFuture<YpTypedId> createObject(String id, CreateObjectRequest<Spec> request) {
        return ypRepository.createObject(id, patchCreateRequest(request));
    }

    @Override
    public CompletableFuture<YpTypedId> createObject(String id, YpTransaction transaction, CreateObjectRequest<Spec> request) {
        return ypRepository.createObject(id, transaction, patchCreateRequest(request));
    }

    @Override
    public CompletableFuture<?> updateObject(String id, UpdateYpObjectRequest<Spec, Status> request) {
        return ypRepository.getTransactionClient()
                .runWithTransaction(transaction -> updateObject(id, transaction, request));
    }

    @Override
    public CompletableFuture<?> updateObject(String id, YpTransaction transaction,
            UpdateYpObjectRequest<Spec, Status> request) {
        UpdateYpObjectRequest.Builder<Spec, Status> patchedRequest = new UpdateYpObjectRequest.Builder<Spec, Status>()
                .setLabels(request.getLabels())
                .setFields(request.getFields())
                .addLabel(epochKey, YTree.integerNode(epochGetter.get()));
        request.getAcl().ifPresent(patchedRequest::setAcl);
        request.getStatus().ifPresent(patchedRequest::setStatus);
        request.getSpec().ifPresent(patchedRequest::setSpec);
        request.getRequestExtender().ifPresent(patchedRequest::setRequestExtender);

        return validateEpoch(transaction, id)
                .thenCompose(x -> ypRepository.updateObject(id, transaction, patchedRequest.build()));
    }

    @Override
    public CompletableFuture<?> removeObject(String id) {
        return ypRepository.getTransactionClient()
                .runWithTransaction(transaction -> removeObject(id, transaction));
    }

    @Override
    public CompletableFuture<?> removeObject(String id, YpTransaction transaction) {
        return validateEpoch(transaction, id).thenCompose(x -> ypRepository.removeObject(id, transaction));
    }

    @Override
    public CompletableFuture<Optional<YpObject<Meta, Spec, Status>>> getObject(String id, Selector selector) {
        return ypRepository.getObject(id, selector);
    }

    @Override
    public CompletableFuture<Optional<YpObject<Meta, Spec, Status>>> getObject(String id, Long timestamp,
            Selector selector) {
        return ypRepository.getObject(id, timestamp, selector);
    }

    @Override
    public CompletableFuture<List<Optional<YpObject<Meta, Spec, Status>>>> getObjects(List<String> ids,
            Selector selector) {
        return ypRepository.getObjects(ids, selector);
    }

    @Override
    public CompletableFuture<List<Optional<YpObject<Meta, Spec, Status>>>> getObjects(List<String> ids,
            Selector selector, Long timestamp) {
        return ypRepository.getObjects(ids, selector, timestamp);
    }

    @Override
    public CompletableFuture<SelectedObjects<Meta, Spec, Status>> selectObjects(
            Selector selector, Map<String, String> labelsForFilter) {
        return ypRepository.selectObjects(selector, labelsForFilter);
    }

    @Override
    public CompletableFuture<SelectedObjects<Meta, Spec, Status>> selectObjects(
            Selector selector, Map<String, String> labelsForFilter, Long endTimestamp) {
        return ypRepository.selectObjects(selector, labelsForFilter, endTimestamp);
    }

    @Override
    public CompletableFuture<WatchedObjects> watchObjects(Long startTimestamp, Duration timeLimit) {
        return ypRepository.watchObjects(startTimestamp, timeLimit);
    }

    @Override
    public CompletableFuture<WatchedObjects> watchObjects(Long startTimestamp, Long endTimestamp, Duration timeLimit) {
        return ypRepository.watchObjects(startTimestamp, endTimestamp, timeLimit);
    }

    @Override
    public CompletableFuture<WatchedObjects> watchObjects(long startTimestamp, Optional<Long> endTimestamp) {
        return ypRepository.watchObjects(startTimestamp, endTimestamp);
    }

    @Override
    public CompletableFuture<Boolean> saveStatus(String id, Status status) {
        return ypRepository.saveStatus(id, new UpdateYpObjectRequest.Builder<Spec, Status>()
                .setStatus(status)
                .setLabels(ImmutableMap.of(epochKey, YTree.integerNode(epochGetter.get())))
                .build());
    }

    @Override
    public CompletableFuture<Boolean> saveStatus(String id, UpdateYpObjectRequest<Spec, Status> request) {
        UpdateYpObjectRequest.Builder<Spec, Status> patchedRequest =  new UpdateYpObjectRequest.Builder<Spec, Status>()
                .setLabels(ImmutableMap.of(epochKey, YTree.integerNode(epochGetter.get())));
        request.getStatus().ifPresent(patchedRequest::setStatus);
        return ypRepository.saveStatus(id, patchedRequest.build());
    }

    @Override
    public CompletableFuture<Long> generateTimestamp() {
        return ypRepository.generateTimestamp();
    }

    @Override
    public CompletableFuture<Set<String>> listAllIds(Optional<Long> timestamp, Map<String, String> labelsForFilter) {
        return ypRepository.listAllIds(timestamp, labelsForFilter);
    }

    @Override
    public CompletableFuture<Set<String>> listIds(Optional<Long> timestamp) {
        return ypRepository.listIds(timestamp);
    }

    private CompletableFuture<?> validateEpoch(YpTransaction tr, String id) {
        return ypRepository.getTransactionClient()
                .startTransaction().thenCompose(transaction ->
                getEpoch(id, transaction)
                        .thenCompose(epoch -> {
                            long currentEpoch = epochGetter.get();
                            if (epoch > currentEpoch) {
                                LOG.info("We lose lock. Object {}:{} epoch {} more then current {}", objectType, id,
                                        epoch,
                                        currentEpoch);
                                throw new RuntimeException(
                                        String.format("Current epoch is %d less than object epoch %d", currentEpoch,
                                                epoch));
                            }

                            return completedFuture(0);
                        }));
    }

    private static String getEpochPath(String epochKey) {
        return String.format("%s/%s", Paths.LABELS, epochKey);
    }

    private CompletableFuture<Long> getEpoch(String id, YpTransaction transaction) {
        Selector selector = new Selector.Builder().withLabels(ImmutableSet.of(epochKey)).build();
        YpGetStatement.Builder statement = YpGetStatement.ysonBuilder(new YpTypedId(id, objectType));
        statement.setTimestamp(transaction.getStartTimestamp());
        statement.addSelector(epochPath);
        return ypRepository.getObject(id, transaction.getStartTimestamp(), selector)
                .thenApply(objectOpt -> objectOpt.orElseThrow().getLabel(epochKey).orElseThrow())
                .thenApply(epoch -> {
                    if (epoch.isIntegerNode()) {
                        return epoch.longValue();
                    } else if (epoch.isEntityNode()) {
                        return epochGetter.get();
                    }
                    throw new RuntimeException(String.format("In %s:%s not integer value %s", id, epochPath, epoch.toString()));
                });
    }

    @Override
    public String toString() {
        return ypRepository.toString();
    }
}
