package ru.yandex.infra.controller.yp;

import java.time.Duration;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableMap;
import com.google.protobuf.Message;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.controller.dto.Acl;
import ru.yandex.infra.controller.dto.SchemaMeta;
import ru.yandex.infra.controller.metrics.GaugeRegistry;
import ru.yandex.infra.controller.util.YsonUtils;
import ru.yandex.inside.yt.kosher.impl.ytree.YTreeProtoUtils;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTreeBuilder;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.library.svnversion.VcsVersion;
import ru.yandex.yp.YpRawObjectService;
import ru.yandex.yp.model.YpErrorCodes;
import ru.yandex.yp.model.YpEvent;
import ru.yandex.yp.model.YpGetManyStatement;
import ru.yandex.yp.model.YpGetStatement;
import ru.yandex.yp.model.YpObjectType;
import ru.yandex.yp.model.YpObjectUpdate;
import ru.yandex.yp.model.YpPayload;
import ru.yandex.yp.model.YpPayloadFormat;
import ru.yandex.yp.model.YpSelectStatement;
import ru.yandex.yp.model.YpSetUpdate;
import ru.yandex.yp.model.YpTransaction;
import ru.yandex.yp.model.YpTypedId;
import ru.yandex.yp.model.YpTypedObject;
import ru.yandex.yp.model.YpWatchObjectsStatement;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static ru.yandex.infra.controller.util.ExceptionUtils.tryExtractYpError;
import static ru.yandex.infra.controller.util.YsonUtils.payloadToYson;
import static ru.yandex.infra.controller.yp.YpUtils.getSelectedPaths;

// Reads and writes objects of specified type with given set of labels
public class LabelBasedRepository<Meta extends SchemaMeta, Spec extends Message, Status extends Message>
        implements YpObjectTransactionalRepository<Meta, Spec, Status> {
    private static final Logger LOG = LoggerFactory.getLogger(LabelBasedRepository.class);
    public static final String ALL_CLUSTERS = "xdc";
    public static final int REQUEST_IDS_PAGE_SIZE = 16384;

    private final Map<String, String> allOperationsLabelsMap;
    private final Map<String, YTreeMapNode> generatedAnnotations;
    private final YpObjectType objectType;
    private final YpRawObjectService ypClient;
    private final YpTransactionClient ypTransactionClient;
    private final ObjectBuilderDescriptor<? extends Message, Meta> typeDescription;
    private final int defaultSelectPageSize;
    private final int defaultWatchPageSize;
    private final String description;
    private volatile Integer currentSelectPageSize;
    private volatile Integer currentWatchPageSize;

    public LabelBasedRepository(YpObjectType objectType, Map<String, String> allOperationsLabelsMap,
            Optional<String> vcsKey, YpRawObjectService ypClient,
            ObjectBuilderDescriptor<? extends Message, Meta> typeDescription,
            int defaultSelectPageSize, int defaultWatchPageSize, GaugeRegistry gaugeRegistry, String cluster)
    {
        this.allOperationsLabelsMap = allOperationsLabelsMap;
        this.generatedAnnotations = getVcsMap(vcsKey);
        this.objectType = objectType;
        this.ypClient = ypClient;
        this.ypTransactionClient = new YpTransactionClientImpl(ypClient);
        this.typeDescription = typeDescription;
        this.defaultSelectPageSize = defaultSelectPageSize;
        this.defaultWatchPageSize = defaultWatchPageSize;
        this.description = cluster + "." + objectType;
        try {
            gaugeRegistry.add(String.format("select_executor.%s.%s.current_page_size", objectType.name(), cluster),
                    () -> currentSelectPageSize);
            gaugeRegistry.add(String.format("watch_executor.%s.%s.current_page_size", objectType.name(), cluster),
                    () -> currentWatchPageSize);
        } catch (IllegalArgumentException exception) {
            LOG.warn("Metrics for object type {} was already added into registry", objectType.name());
        }
    }

    public LabelBasedRepository(YpObjectType objectType, Map<String, String> allOperationsLabelsMap,
            Optional<String> vcsKey, YpRawObjectService ypClient,
            ObjectBuilderDescriptor<? extends Message, Meta> typeDescription,
            int defaultPageSize, GaugeRegistry gaugeRegistry, String cluster)
    {
        this(objectType, allOperationsLabelsMap, vcsKey, ypClient, typeDescription, defaultPageSize, defaultPageSize,
                gaugeRegistry, cluster);
    }

    public LabelBasedRepository(YpObjectType objectType, Map<String, String> allOperationsLabelsMap,
                                Optional<String> vcsKey, YpRawObjectService ypClient,
                                ObjectBuilderDescriptor<? extends Message, Meta> typeDescription,
                                int defaultPageSize, GaugeRegistry gaugeRegistry)
    {
        this(objectType, allOperationsLabelsMap, vcsKey, ypClient, typeDescription, defaultPageSize, defaultPageSize,
                gaugeRegistry, ALL_CLUSTERS);
    }

    public LabelBasedRepository(YpObjectType objectType, Map<String, String> allOperationsLabelsMap,
            Optional<String> vcsKey, YpRawObjectService ypClient,
            ObjectBuilderDescriptor<? extends Message, Meta> typeDescription,
            int defaultPageSize,int defaultWatchPageSize, GaugeRegistry gaugeRegistry)
    {
        this(objectType, allOperationsLabelsMap, vcsKey, ypClient, typeDescription, defaultPageSize, defaultWatchPageSize,
                gaugeRegistry, ALL_CLUSTERS);
    }

    @Override
    public YpTransactionClient getTransactionClient() {
        return ypTransactionClient;
    }

    @Override
    public YpObjectType getObjectType() {
        return objectType;
    }

    @Override
    public CompletableFuture<YpTypedId> createObject(String id, CreateObjectRequest<Spec> request) {
        return doCreateObject(id, Optional.empty(), request);
    }

    @Override
    public CompletableFuture<YpTypedId> createObject(String id, YpTransaction transaction, CreateObjectRequest<Spec> request) {
        return doCreateObject(id, Optional.of(transaction), request);
    }

    @Override
    public CompletableFuture<?> updateObject(String id, UpdateYpObjectRequest<Spec, Status> request) {
        return doUpdateObject(id, Optional.empty(), request);
    }

    @Override
    public CompletableFuture<?> updateObject(String id, YpTransaction transaction,
            UpdateYpObjectRequest<Spec, Status> request) {
        return doUpdateObject(id, Optional.of(transaction), request);
    }

    @Override
    public CompletableFuture<?> removeObject(String id) {
        return doRemoveObject(id, Optional.empty());
    }

    @Override
    public CompletableFuture<?> removeObject(String id, YpTransaction transaction) {
        return doRemoveObject(id, Optional.of(transaction));
    }

    @Override
    public CompletableFuture<Optional<YpObject<Meta, Spec, Status>>> getObject(String id, Selector selector) {
        return doGetObject(id, Optional.empty(), selector);
    }

    @Override
    public CompletableFuture<List<Optional<YpObject<Meta, Spec, Status>>>> getObjects(
            List<String> ids, Selector selector) {
        return doGetObjects(ids, Optional.empty(), selector);
    }

    @Override
    public CompletableFuture<Optional<YpObject<Meta, Spec, Status>>> getObject(String id,
            Long timestamp, Selector selector) {
        return doGetObject(id, Optional.of(timestamp), selector);
    }

    @Override
    public CompletableFuture<List<Optional<YpObject<Meta, Spec, Status>>>> getObjects(
            List<String> ids, Selector selector, Long timestamp) {
        return doGetObjects(ids, Optional.of(timestamp), selector);
    }

    private CompletableFuture<YpTypedId> doCreateObject(String id, Optional<YpTransaction> transactionOpt,
            CreateObjectRequest<Spec> request) {
        Map<String, Object> labels = new HashMap<>(allOperationsLabelsMap);
        labels.putAll(request.getLabels());
        YTreeBuilder builder = YTree.mapBuilder()
                .key("spec").value(YTreeProtoUtils.marshal(request.getSpec(), true))
                .key("labels").value(labels)
                .key("annotations").value(generatedAnnotations)//stagectl_vcs={"svn_tag"="";"svn_revision"=-1;"svn_branch"=""}
                .key("meta").beginMap()
                .key("id").value(id);
        request.getAcl().ifPresent(acl -> builder.key("acl").value(aclToYson(acl)));
        request.getSpecificMetaFields().forEach((key, value) -> builder.key(key).value(value));
        builder.endMap();

        YpTypedObject object = new YpTypedObject(objectType, YsonUtils.toYsonPayload(builder.buildMap()));
        return transactionOpt
                .map(transaction -> ypClient.createObject(object, transaction))
                .orElseGet(() -> ypClient.createObject(object));
    }

    private CompletableFuture<?> doUpdateObject(String id, Optional<YpTransaction> transactionOpt,
            UpdateYpObjectRequest<Spec, Status> request) {
        YpObjectUpdate.Builder builder = YpObjectUpdate.builder(toTypedId(id));
        builder.addAttributeTimestampPrerequisites(request.getPrerequisites());

        //stagectl_epoch
        request.getLabels().forEach((name, node) -> {
            if (objectType != YpObjectType.STAGE) {
                LOG.debug("{}: Updating label {} = {}", id, name, node);
                builder.addSetUpdate(new YpSetUpdate(Paths.LABELS + Paths.DELIMITER + name, node, YsonUtils::toYsonPayload));
            }
            else {
                LOG.debug("{}: Updating annotation {} = {}", id, name, node);
                builder.addSetUpdate(new YpSetUpdate(Paths.ANNOTATIONS + Paths.DELIMITER + name, node, YsonUtils::toYsonPayload));
            }
        });

        //stagectl_vcs={"svn_tag"="";"svn_revision"=-1;"svn_branch"=""}
        generatedAnnotations.forEach((name, value) -> {
            YTreeNode node = YTree.builder().value(value).build();
            LOG.debug("{}: Updating annotation {} = {}", id, name, value);
            builder.addSetUpdate(new YpSetUpdate(Paths.ANNOTATIONS + Paths.DELIMITER + name, node, YsonUtils::toYsonPayload));
        });

        //No reason to update labels from 'allOperationsLabelsMap', i.e. from config: {deploy_engine, created_by, stage_tag...}

        request.getAcl().ifPresent(
                acl -> builder.addSetUpdate(new YpSetUpdate(Paths.ACL, aclToYson(acl), YsonUtils::toYsonPayload)));
        request.getSpec().ifPresent(spec -> builder.addSetUpdate(new YpSetUpdate(Paths.SPEC,
                YTreeProtoUtils.marshal(spec, true), YsonUtils::toYsonPayload)));

        request.getStatus().ifPresent(status -> builder.addSetUpdate(new YpSetUpdate(Paths.STATUS,
                YTreeProtoUtils.marshal(status, true), YsonUtils::toYsonPayload)));

        request.getFields().forEach((field, node) -> builder.addSetUpdate(new YpSetUpdate(
                field,
                node,
                YsonUtils::toYsonPayload
        )));

        request.getRequestExtender().ifPresent(extender -> extender.accept(builder));

        return transactionOpt
                .map(transaction -> ypClient.updateObject(builder.build(), transaction))
                .orElseGet(() -> ypClient.updateObject(builder.build()));
    }

    private CompletableFuture<?> doRemoveObject(String id, Optional<YpTransaction> transactionOpt) {
        return transactionOpt
                .map(transaction -> ypClient.removeObject(toTypedId(id), transaction))
                .orElseGet(() -> ypClient.removeObject(toTypedId(id)));
    }

    private CompletableFuture<Optional<YpObject<Meta, Spec, Status>>> doGetObject(String id,
            Optional<Long> timestampOpt, Selector selector) {
        YpGetStatement.Builder builder = YpGetStatement.ysonBuilder(toTypedId(id))
                .setFetchTimestamps(selector.hasFetchTimestamps());
        timestampOpt.ifPresent(builder::setTimestamp);

        getSelectedPaths(selector).forEach(builder::addSelector);
        return ypClient.getObject(builder.build(), payloads -> parseObject(selector, payloads.iterator()))
                .handle((result, error) -> {
                    if (error != null) {
                        return tryRecoverFromNoSuchObject(error, Optional.empty());
                    }
                    return Optional.of(result);
                });
    }

    private CompletableFuture<List<Optional<YpObject<Meta, Spec, Status>>>> doGetObjects(List<String> ids,
            Optional<Long> timestampOpt, Selector selector) {
        YpGetManyStatement.Builder builder = YpGetManyStatement.ysonBuilder(objectType)
                .setFetchTimestamps(selector.hasFetchTimestamps())
                .setFetchValues(selector.hasFetchValues())
                .addIds(ids)
                .setIgnoreNonexistentObjects(true);
        timestampOpt.ifPresent(builder::setTimestamp);
        getSelectedPaths(selector).forEach(builder::addSelector);

        return ypClient.getObjects(builder.build(), payloads -> {
            Iterator<YpPayload> iterator = payloads.iterator();
            if (!iterator.hasNext()) {
                return Optional.empty();
            }
            final YpObject<Meta, Spec, Status> ypObject = parseObject(selector, iterator);

            if (selector.hasAllLabels()) {
                //GetObjects is usually used after watchObjects.
                //but watchObjects have no label filters at all, so we should filter objects manually
                for (Map.Entry<String, String> entry : allOperationsLabelsMap.entrySet()) {
                    if (ypObject.getLabel(entry.getKey()).isEmpty() ||
                            !ypObject.getLabel(entry.getKey()).get().stringValue().equals(entry.getValue())) {
                        return Optional.empty();
                    }
                }
            }

            return Optional.of(ypObject);
        });
    }

    @Override
    public YpRawObjectService getRawClient() {
        return ypClient;
    }

    @Override
    public CompletableFuture<Set<String>> listIds(Optional<Long> timestamp) {
        return listAllIds(timestamp, allOperationsLabelsMap);
    }

    @Override
    public CompletableFuture<Set<String>> listAllIds(Optional<Long> timestamp, Map<String, String> labelsForFilter) {
        YpSelectStatement.Builder builder = YpSelectStatement.builder(objectType, YpPayloadFormat.YSON)
                .addSelector(Paths.ID);
        if(!labelsForFilter.isEmpty()) {
            builder.setFilter(YpUtils.getFilterQuery(null, labelsForFilter));
        }
        timestamp.ifPresent(builder::setTimestamp);
        return YpRequestWithPaging.selectObjects(ypClient, REQUEST_IDS_PAGE_SIZE, builder, payloads -> payloadToYson(payloads.get(0)).stringValue())
                .thenApply(HashSet::new);
    }

    @Override
    public CompletableFuture<SelectedObjects<Meta, Spec, Status>> selectObjects(Selector selector,
            Map<String, String> labelsForFilter) {
        return selectObjects(selector, labelsForFilter, Optional.empty());
    }

    @Override
    public CompletableFuture<SelectedObjects<Meta, Spec, Status>> selectObjects(Selector selector,
            Map<String, String> labelsForFilter, Long endTimestamp) {
        return selectObjects(selector, labelsForFilter, Optional.of(endTimestamp));
    }

    private CompletableFuture<SelectedObjects<Meta, Spec, Status>> selectObjects(
            Selector selector, Map<String, String> labelsForFilter, Optional<Long> timestamp) {
        Map<String, String> unionLabels = new HashMap<>(allOperationsLabelsMap);
        unionLabels.putAll(labelsForFilter);

        SelectObjectsExecutor<Meta, Spec, Status> execution = new SelectObjectsExecutor<>(ypClient, selector,
                unionLabels, timestamp, defaultSelectPageSize, objectType, this::parseObject);
        currentSelectPageSize = execution.getCurrentPageSize();
        return execution.run();
    }

    @Override
    public CompletableFuture<WatchedObjects> watchObjects(long startTimestamp, Optional<Long> endTimestamp) {
        YpWatchObjectsStatement.Builder builder = YpWatchObjectsStatement.builder(objectType, startTimestamp);
        endTimestamp.ifPresent(builder::setTimestamp);

        //Don't limit page size. If YP contains too many updates to fit into single grpc message ->
        //just handle exception and reload all objects with selectObjects request instead of watchObject

        return ypClient.watchObjects(builder.build())
                .thenApply(watchedObjects -> new WatchedObjects(
                        watchedObjects.getEvents().stream()
                                .collect(Collectors.<YpEvent, String>groupingBy(ypEvent -> ypEvent.getId().getId())),
                        watchedObjects.getTimestamp()));
    }

    @Override
    public CompletableFuture<WatchedObjects> watchObjects(Long startTimestamp, Duration timeLimit) {
        WatchObjectsExecutor executor = new WatchObjectsExecutor(ypClient, startTimestamp, ypClient.generateTimestamp(),
                defaultWatchPageSize, objectType, timeLimit);
        currentWatchPageSize = executor.getCurrentPageSize();
        return executor.run();
    }

    @Override
    public CompletableFuture<Long> generateTimestamp() {
        return ypClient.generateTimestamp();
    }

    @Override
    public CompletableFuture<WatchedObjects> watchObjects(Long startTimestamp, Long endTimestamp, Duration timeLimit) {
        WatchObjectsExecutor executor = new WatchObjectsExecutor(ypClient, startTimestamp,
                completedFuture(endTimestamp), defaultWatchPageSize, objectType, timeLimit);
        currentWatchPageSize = executor.getCurrentPageSize();
        return executor.run();
    }

    @Override
    public CompletableFuture<Boolean> saveStatus(String id, UpdateYpObjectRequest<Spec, Status> request) {
        return doUpdateObject(id, Optional.empty(), request)
                .handle((result, error) -> {
                    if (error != null) {
                        return tryRecoverFromNoSuchObject(error, false);
                    }
                    return true;
                });
    }

    private static YTreeNode aclToYson(Acl acl) {
        YTreeBuilder builder = new YTreeBuilder();
        builder.beginList();
        acl.getEntries().forEach(entry -> builder.value(YTreeProtoUtils.marshal(entry, true)));
        builder.endList();
        return builder.build();
    }

    private static <T> T tryRecoverFromNoSuchObject(Throwable error, T ifRecovered) {
        if (tryExtractYpError(error).filter(ypError -> ypError.getCode() == YpErrorCodes.NO_SUCH_OBJECT).isPresent()) {
            return ifRecovered;
        }
        Throwables.throwIfUnchecked(error);
        throw new RuntimeException(error);
    }

    private YpObject<Meta, Spec, Status> parseObject(Selector selector, Iterator<YpPayload> payloads) {
        // order of payloads is consistent with output of getSelectedPaths()
        YpObject.Builder<Meta, Spec, Status> builder = new YpObject.Builder<>();
        if (selector.hasMeta()) {
            YpPayload metaPayload = payloads.next();
            if (selector.hasFetchValues()) {
                builder.setMeta(typeDescription.createMeta(
                        protoBuilder -> YTreeProtoUtils.unmarshal(payloadToYson(metaPayload), protoBuilder)));
            }
            if (selector.hasFetchTimestamps()) {
                builder.setMetaTimestamp(metaPayload.getTimestamp().orElseThrow());
            }
        }
        if (selector.hasSpec()) {
            YpPayload specPayload = payloads.next();
            if (selector.hasFetchValues()) {
                builder.setSpec((Spec) YTreeProtoUtils.unmarshal(payloadToYson(specPayload), typeDescription.createSpecBuilder()));
            }
            if (selector.hasFetchTimestamps()) {
                builder.setSpecTimestamp(specPayload.getTimestamp().orElseThrow());
            }
        }
        if (selector.hasStatus()) {
            YpPayload statusPayload = payloads.next();
            if (selector.hasFetchValues()) {
                builder.setStatus((Status) YTreeProtoUtils.unmarshal(payloadToYson(statusPayload), typeDescription.createStatusBuilder()));
            }
            if (selector.hasFetchTimestamps()) {
                builder.setStatusTimestamp(statusPayload.getTimestamp().orElseThrow());
            }
        }
        if (selector.hasAllLabels()) {
            YTreeNode payload = payloadToYson(payloads.next());
            Map<String, YTreeNode> attrs = payload.asMap();
            builder.setLabels(attrs);
        }

        selector.getAnnotations().forEach(annotation -> {
            YTreeNode node = payloadToYson(payloads.next());
            if (!node.isEntityNode()) {
                builder.putAnnotation(annotation, node.stringValue());
            }
        });
        selector.getRequiredLabelKeys().forEach(label -> builder.putLabel(label, payloadToYson(payloads.next())));

        return builder.build();
    }

    private YpTypedId toTypedId(String id) {
        return new YpTypedId(id, objectType);
    }

    private static Map<String, YTreeMapNode> getVcsMap(Optional<String> prefixOptional) {
        if (prefixOptional.isEmpty()) {
            return Collections.emptyMap();
        }
        String prefix = prefixOptional.get();
        if (prefix.isEmpty()) {
            throw new IllegalArgumentException("Vcs label prefix is empty");
        }
        if (prefix.contains(Paths.DELIMITER)) {
            throw new IllegalArgumentException(String.format("Vcs label prefix %s contains delimiter - not supported", prefix));
        }
        VcsVersion vcsVersion = new VcsVersion(LabelBasedRepository.class);
        YTreeMapNode mapNode = new YTreeBuilder()
                .beginMap()
                    .key("svn_revision").value(vcsVersion.getProgramSvnRevision())
                    .key("svn_branch").value(vcsVersion.getBranch())
                    .key("svn_tag").value(vcsVersion.getTag())
                .buildMap();

        return ImmutableMap.of(prefix, mapNode);
    }

    @Override
    public String toString() {
        return description;
    }
}
