package ru.yandex.infra.controller.yp;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ExecutionException;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;

import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.infra.controller.util.ExceptionUtils;
import ru.yandex.yp.model.YpObjectType;

public class ObjectsPaginationExecutor<T> {
    private static final Logger LOG = LoggerFactory.getLogger(ObjectsPaginationExecutor.class);

    private final int defaultPageSize;
    private final YpObjectType objectType;
    private final Map<String, T> accumulator = new HashMap<>();

    private BiFunction<Long, Optional<String>, CompletableFuture<PageResult>> getNextPageFunc;
    private BiConsumer<Map<String, T>, Map<String, T>> updateAccumulatorFunc;
    private int currentPageSize;
    private Optional<String> previousId = Optional.empty();

    public ObjectsPaginationExecutor(int defaultPageSize, YpObjectType objectType) {
        this.defaultPageSize = defaultPageSize;
        this.objectType = objectType;
        this.currentPageSize = defaultPageSize;
        this.getNextPageFunc = null;
        this.updateAccumulatorFunc = null;
    }

    public void setGetNextFunction(BiFunction<Long, Optional<String>, CompletableFuture<PageResult>> getNextPageFunc) {
        this.getNextPageFunc = getNextPageFunc;
    }

    public void setUpdateAccumulatorFunction(BiConsumer<Map<String, T>, Map<String, T>> updateAccumulatorFunc) {
        this.updateAccumulatorFunc = updateAccumulatorFunc;
    }

    public int getCurrentPageSize() {
        return currentPageSize;
    }

    protected class PageResult {
        final private Map<String, T> objects;
        final private Optional<String> continuationToken;

        public PageResult(Map<String, T> objects, Optional<String> continuationToken) {
            this.objects = objects;
            this.continuationToken = continuationToken;
        }

        public Map<String, T> getObjects() {
            return objects;
        }

        public Optional<String> getContinuationToken() {
            return continuationToken;
        }
    }

    protected Map<String, T> getObjects(Long timestamp, Optional<String> continuationTokenOpt) {
        try {
            PageResult pageResult = getNextPageFunc.apply(timestamp, continuationTokenOpt).get();

            Map<String, T> idsToObjects = pageResult.getObjects();
            updateAccumulatorFunc.accept(accumulator, idsToObjects);

            if (idsToObjects.size() < currentPageSize) {
                return accumulator;
            }
            currentPageSize = defaultPageSize;
            previousId = idsToObjects.keySet().stream().max(String::compareTo);
            return getObjects(timestamp, pageResult.getContinuationToken());

        } catch (CompletionException | ExecutionException | InterruptedException error) {
            // DEPLOY-2209 - try smaller batches if failed because of message size
            // DEPLOY-5433 - fix trying smaller batches
            if (currentPageSize == 1) {
                LOG.error("Listing of {} with page size {} failed after {}, will fail utterly and entirely: ",
                        objectType.getType(), currentPageSize, previousId, error);
                throw new ObjectsPaginationExcecutorException(error);
            } else if (ExceptionUtils.tryExtractExceptionOfType(error, StatusRuntimeException.class)
                    .filter(e -> e.getStatus().getCode() == Status.Code.RESOURCE_EXHAUSTED)
                    .isPresent()) {
                int newPageSize = currentPageSize / 2;
                LOG.warn("Listing of {} with page size {} failed after {}, will try new page size {}, error: {}",
                        objectType.getType(), currentPageSize, previousId, newPageSize, error.getMessage());
                currentPageSize = newPageSize;
                return getObjects(timestamp, continuationTokenOpt);
            }
            throw new ObjectsPaginationExcecutorException(error);
        }
    }
}



