package ru.yandex.infra.controller.yp;

import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.yp.YpPayloadDeserializer;
import ru.yandex.yp.YpRawObjectService;
import ru.yandex.yp.model.YpCheckObjectPermissions;
import ru.yandex.yp.model.YpCheckedObjectPermissions;
import ru.yandex.yp.model.YpGetManyStatement;
import ru.yandex.yp.model.YpGetObjectAccessAllowedFor;
import ru.yandex.yp.model.YpGetStatement;
import ru.yandex.yp.model.YpGetUserAccessAllowedTo;
import ru.yandex.yp.model.YpObjectAccessAllowedFor;
import ru.yandex.yp.model.YpObjectHistory;
import ru.yandex.yp.model.YpObjectHistoryStatement;
import ru.yandex.yp.model.YpObjectUpdate;
import ru.yandex.yp.model.YpSelectStatement;
import ru.yandex.yp.model.YpSelectedObjects;
import ru.yandex.yp.model.YpTransaction;
import ru.yandex.yp.model.YpTypedId;
import ru.yandex.yp.model.YpTypedObject;
import ru.yandex.yp.model.YpUserAccessAllowedTo;
import ru.yandex.yp.model.YpWatchObjectsStatement;
import ru.yandex.yp.model.YpWatchedObjects;

public class ReadonlyYpRawObjectService implements YpRawObjectService {
    private static final Logger LOG = LoggerFactory.getLogger(ReadonlyYpRawObjectService.class);

    private final YpRawObjectService yp;

    public ReadonlyYpRawObjectService(YpRawObjectService yp) {
        this.yp = yp;
    }

    @Override
    public CompletableFuture<Long> generateTimestamp() {
        return yp.generateTimestamp();
    }

    @Override
    public CompletableFuture<YpTransaction> startTransaction() {
        return yp.startTransaction();
    }

    @Override
    public CompletableFuture<Long> commitTransaction(YpTransaction transaction) {
        return yp.commitTransaction(transaction);
    }

    @Override
    public CompletableFuture<Void> abortTransaction(YpTransaction transaction) {
        return yp.abortTransaction(transaction);
    }

    @Override
    public CompletableFuture<YpTypedId> createObject(YpTypedObject object) {
        LOG.info("[readonly mode] Operation skipped: createObject {}", object);
        return CompletableFuture.completedFuture(new YpTypedId(null, object.getType(), Optional.of("")));
    }

    @Override
    public CompletableFuture<YpTypedId> createObject(YpTypedObject object, YpTransaction transaction) {
        return createObject(object);
    }

    @Override
    public CompletableFuture<List<YpTypedId>> createObjects(List<YpTypedObject> objects) {
        LOG.info("[readonly mode] Operation skipped: createObjects ({} items)", objects.size());
        List<YpTypedId> result = objects.stream()
                .map(o -> new YpTypedId(null, o.getType()))
                .collect(Collectors.toList());
        return CompletableFuture.completedFuture(result);
    }

    @Override
    public CompletableFuture<List<YpTypedId>> createObjects(List<YpTypedObject> objects, YpTransaction transaction) {
        return createObjects(objects);
    }

    @Override
    public CompletableFuture<Void> removeObject(YpTypedId id) {
        LOG.info("[readonly mode] Operation skipped: removeObject {}", id);
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public CompletableFuture<Void> removeObject(YpTypedId id, YpTransaction transaction) {
        return removeObject(id);
    }

    @Override
    public CompletableFuture<Void> removeObjects(List<YpTypedId> ids) {
        LOG.info("[readonly mode] Operation skipped: removeObjects {}", ids);
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public CompletableFuture<Void> removeObjects(List<YpTypedId> ids, YpTransaction transaction) {
        return removeObjects(ids);
    }

    @Override
    public CompletableFuture<Void> updateObject(YpObjectUpdate update) {
        LOG.info("[readonly mode] Operation skipped: updateObject {}", update);
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public CompletableFuture<Void> updateObject(YpObjectUpdate update, YpTransaction transaction) {
        return updateObject(update);
    }

    @Override
    public CompletableFuture<Void> updateObjects(List<YpObjectUpdate> updates) {
        List<YpTypedId> ids = updates.stream().map(YpObjectUpdate::getId).collect(Collectors.toList());
        LOG.info("[readonly mode] Operation skipped: updateObjects {}", ids);
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public CompletableFuture<Void> updateObjects(List<YpObjectUpdate> updates, YpTransaction transaction) {
        return updateObjects(updates);
    }

    @Override
    public <T> CompletableFuture<T> getObject(YpGetStatement statement, YpPayloadDeserializer<T> payloadDeserializer) {
        return yp.getObject(statement, payloadDeserializer);
    }

    @Override
    public <T> CompletableFuture<List<T>> getObjects(YpGetManyStatement statement, YpPayloadDeserializer<T> payloadDeserializer) {
        return yp.getObjects(statement, payloadDeserializer);
    }

    @Override
    public <T> CompletableFuture<YpSelectedObjects<T>> selectObjects(YpSelectStatement statement, YpPayloadDeserializer<T> payloadDeserializer) {
        return yp.selectObjects(statement, payloadDeserializer);
    }

    @Override
    public CompletableFuture<List<YpCheckedObjectPermissions>> checkObjectPermissions(List<YpCheckObjectPermissions> permissions) {
        return yp.checkObjectPermissions(permissions);
    }

    @Override
    public CompletableFuture<List<YpCheckedObjectPermissions>> checkObjectPermissions(List<YpCheckObjectPermissions> permissions, long timestamp) {
        return yp.checkObjectPermissions(permissions, timestamp);
    }

    @Override
    public CompletableFuture<List<YpObjectAccessAllowedFor>> getObjectAccessAllowedFor(List<YpGetObjectAccessAllowedFor> requests) {
        return yp.getObjectAccessAllowedFor(requests);
    }

    @Override
    public CompletableFuture<List<YpObjectAccessAllowedFor>> getObjectAccessAllowedFor(List<YpGetObjectAccessAllowedFor> requests, long timestamp) {
        return yp.getObjectAccessAllowedFor(requests, timestamp);
    }

    @Override
    public CompletableFuture<List<YpUserAccessAllowedTo>> getUserAccessAllowedTo(List<YpGetUserAccessAllowedTo> requests) {
        return yp.getUserAccessAllowedTo(requests);
    }

    @Override
    public CompletableFuture<YpWatchedObjects> watchObjects(YpWatchObjectsStatement statement) {
        return yp.watchObjects(statement);
    }

    @Override
    public <T> CompletableFuture<YpObjectHistory<T>> selectObjectHistory(YpObjectHistoryStatement statement, YpPayloadDeserializer<T> payloadDeserializer) {
        return yp.selectObjectHistory(statement, payloadDeserializer);
    }
}
