package ru.yandex.infra.controller.yp;

import java.util.Iterator;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.BiFunction;

import com.google.protobuf.Message;
import one.util.streamex.StreamEx;

import ru.yandex.bolts.collection.Try;
import ru.yandex.infra.controller.dto.SchemaMeta;
import ru.yandex.infra.controller.util.YsonUtils;
import ru.yandex.yp.YpRawObjectService;
import ru.yandex.yp.model.YpObjectType;
import ru.yandex.yp.model.YpPayload;
import ru.yandex.yp.model.YpPayloadFormat;
import ru.yandex.yp.model.YpSelectStatement;

import static ru.yandex.infra.controller.yp.YpUtils.getSelectedPaths;

class SelectObjectsExecutor<Meta extends SchemaMeta, Spec extends Message, Status extends Message>
        extends ObjectsPaginationExecutor<Try<YpObject<Meta, Spec, Status>>> {
    private final YpRawObjectService ypClient;
    private final Selector selector;
    private final String filter;

    private final Optional<Long> timestamp;
    private final YpObjectType objectType;
    private final BiFunction<Selector, Iterator<YpPayload>, YpObject<Meta, Spec, Status>> parseObjectFunc;

    public SelectObjectsExecutor(YpRawObjectService ypClient, Selector selector, Map<String, String> labelsForFilter,
                                 Optional<Long> timestamp, int defaultPageSize, YpObjectType objectType,
                                 BiFunction<Selector, Iterator<YpPayload>, YpObject<Meta, Spec, Status>> parseObjectFunc)
    {
        super(defaultPageSize, objectType);
        super.setGetNextFunction(this::getNextPage);
        super.setUpdateAccumulatorFunction(this::updateStoredObjects);
        this.ypClient = ypClient;
        this.selector = selector;
        this.filter = YpUtils.getFilterQuery(selector.getFilter(), labelsForFilter);
        this.timestamp = timestamp;
        this.objectType = objectType;
        this.parseObjectFunc = parseObjectFunc;
    }

    public CompletableFuture<SelectedObjects<Meta, Spec, Status>> run() {
        return ypClient.generateTimestamp()
                .thenApply(ypTimestamp -> {
                    Long t = timestamp.orElse(ypTimestamp);
                    Map<String, Try<YpObject<Meta,Spec,Status>>> objects = getObjects(t, Optional.empty());
                    return new SelectedObjects<>(objects, t);
                });
    }

    private CompletableFuture<PageResult> getNextPage(Long timestamp, Optional<String> continuationTokenOpt) {
        YpSelectStatement.Builder statusMetaStatementBuilder =
                YpSelectStatement.builder(objectType, YpPayloadFormat.YSON)
                        .setFetchTimestamps(selector.hasFetchTimestamps())
                        .setFetchValues(selector.hasFetchValues())
                        .setLimit(super.getCurrentPageSize())
                        .setTimestamp(timestamp);

        continuationTokenOpt.ifPresent(statusMetaStatementBuilder::setContinuationToken);
        statusMetaStatementBuilder.addSelector(Paths.ID);
        getSelectedPaths(selector).forEach(statusMetaStatementBuilder::addSelector);

        if (!filter.isEmpty()) {
            statusMetaStatementBuilder.setFilter(filter);
        }
        return ypClient.selectObjects(statusMetaStatementBuilder.build(), payloads -> payloads)
                .thenApply(selectedObjects -> new PageResult(
                        StreamEx.of(selectedObjects.getResults()).toMap(YsonUtils::parseId,
                                payloads -> {
                                    Iterator<YpPayload> payloadIterator = payloads.iterator();
                                    payloadIterator.next();
                                    return Try.tryCatchException(() -> parseObjectFunc.apply(selector, payloadIterator));
                                }),
                        selectedObjects.getContinuationToken())
                );
    }

    private void updateStoredObjects(Map<String, Try<YpObject<Meta, Spec, Status>>> storedObjects,
            Map<String, Try<YpObject<Meta, Spec, Status>>> newObjects) {
        storedObjects.putAll(newObjects);
    }
}
