package ru.yandex.infra.controller.yp;

import java.util.Collections;
import java.util.Objects;
import java.util.Set;

import com.google.common.base.MoreObjects;

import ru.yandex.misc.lang.StringUtils;

// Add new paths as required
public class Selector {

    private final boolean spec;
    private final boolean meta;
    private final boolean status;
    private final boolean labels;
    private final Set<String> labelsKeys;
    private final Set<String> annotations;
    private final boolean fetchTimestamps;
    private final boolean fetchValues;
    private final String filter;

    private Selector(boolean spec,
                     boolean meta,
                     boolean status,
                     boolean labels,
                     Set<String> labelsKeys,
                     Set<String> annotations,
                     boolean fetchTimestamps,
                     boolean fetchValues,
                     String filter) {
        this.spec = spec;
        this.meta = meta;
        this.status = status;
        this.labels = labels;
        this.labelsKeys = labelsKeys;
        this.annotations = annotations;
        this.fetchTimestamps = fetchTimestamps;
        this.fetchValues = fetchValues;
        this.filter = filter;
    }

    public Selector withLabels() {
        return new Selector(spec, meta, status, true, labelsKeys, annotations, fetchTimestamps, fetchValues, filter);
    }

    public static class Builder {
        private boolean spec = false;
        private boolean meta = false;
        private boolean status = false;
        private boolean labels = false;
        private Set<String> labelKeys = Collections.emptySet();
        private Set<String> annotations = Collections.emptySet();
        private boolean fetchTimestamps = false;
        private boolean fetchValues = true;
        private String filter = null;

        public Builder withSpecAndTimestamps() {
            return withSpec().withTimestamps();
        }

        public Builder withSpec() {
            spec = true;
            return this;
        }

        public Builder withMeta() {
            meta = true;
            return this;
        }

        public Builder withStatus() {
            status = true;
            return this;
        }

        public Builder withAnnotations(Set<String> annotations) {
            this.annotations = annotations;
            return this;
        }

        public Builder withLabels() {
            this.labels = true;
            return this;
        }

        public Builder withLabels(Set<String> labels) {
            this.labelKeys = labels;
            return this;
        }

        public Builder withFilter(String filter) {
            this.filter = filter;
            return this;
        }

        public Builder withTimestamps() {
            fetchTimestamps = true;
            return this;
        }

        public Builder withoutValues() {
            fetchValues = false;
            return this;
        }

        public Selector build() {
            return new Selector(spec, meta, status, labels, labelKeys, annotations, fetchTimestamps, fetchValues, filter);
        }
    }

    public boolean hasSpec() {
        return spec;
    }

    public boolean hasMeta() {
        return meta;
    }

    public boolean hasStatus() {
        return status;
    }

    public boolean hasAllLabels() {
        return labels;
    }

    public boolean hasFetchTimestamps() {
        return fetchTimestamps;
    }

    public boolean hasFetchValues() {
        return fetchValues;
    }

    public Set<String> getAnnotations() {
        return annotations;
    }

    public Set<String> getRequiredLabelKeys() {
        return labelsKeys;
    }

    public String getFilter() {
        return filter;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (!(o instanceof Selector)) return false;
        Selector selector = (Selector) o;
        return spec == selector.spec &&
                meta == selector.meta &&
                status == selector.status &&
                labels == selector.labels &&
                annotations.equals(selector.annotations) &&
                labelsKeys == selector.labelsKeys &&
                fetchTimestamps == selector.fetchTimestamps &&
                fetchValues == selector.fetchValues &&
                StringUtils.equals(filter, selector.filter);
    }

    @Override
    public int hashCode() {
        return Objects.hash(spec, meta, status, labels, annotations, labelsKeys, fetchTimestamps, fetchValues, filter);
    }


    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("spec", spec)
                .add("meta", meta)
                .add("status", status)
                .add("labels", labels)
                .add("annotations", annotations)
                .add("labelKeys", labelsKeys)
                .add("fetchTimestamps", fetchTimestamps)
                .add("fetchValues", fetchValues)
                .add("filter", filter)
                .toString();
    }
}
