package ru.yandex.infra.controller.yp;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;

import com.google.common.base.MoreObjects;
import com.google.protobuf.Message;

import ru.yandex.infra.controller.dto.Acl;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yp.model.YpAttributeTimestampPrerequisite;
import ru.yandex.yp.model.YpObjectUpdate;

// Type-safe update request that is aware of common YP objects structure.
// Add new fields as needed.
public class UpdateYpObjectRequest<Spec extends Message, Status extends Message> {
    private final Optional<Spec> spec;
    private final Optional<Acl> acl;
    private final Optional<Status> status;
    private final Map<String, YTreeNode> labels;
    private final Map<String, YTreeNode> fields;
    private final List<YpAttributeTimestampPrerequisite> prerequisites;
    private final Optional<Consumer<YpObjectUpdate.Builder>> requestExtender;

    private UpdateYpObjectRequest(Optional<Spec> spec, Optional<Acl> acl, Optional<Status> status,
            Map<String, YTreeNode> labels,
            Map<String, YTreeNode> fields,
            List<YpAttributeTimestampPrerequisite> prerequisites,
            Optional<Consumer<YpObjectUpdate.Builder>> requestExtender)
    {
        this.spec = spec;
        this.acl = acl;
        this.status = status;
        this.labels = labels;
        this.fields = fields;
        this.prerequisites = prerequisites;
        this.requestExtender = requestExtender;
    }

    public static class Builder<Spec extends Message, Status extends Message> {
        private Optional<Spec> spec = Optional.empty();
        private Optional<Acl> acl = Optional.empty();
        private Optional<Status> status = Optional.empty();
        private Map<String, YTreeNode> labels = new HashMap<>();
        private Map<String, YTreeNode> fields = new HashMap<>();
        private List<YpAttributeTimestampPrerequisite> prerequisites = new ArrayList<>();
        private Optional<Consumer<YpObjectUpdate.Builder>> requestExtender = Optional.empty();

        public Optional<Spec> getSpecOpt() {
            return spec;
        }

        public Builder<Spec, Status> setSpec(Spec spec) {
            this.spec = Optional.of(spec);
            return this;
        }

        public Builder<Spec, Status> setAcl(Acl acl) {
            this.acl = Optional.of(acl);
            return this;
        }

        public Builder<Spec, Status> setStatus(Status status) {
            this.status = Optional.of(status);
            return this;
        }

        public Map<String, YTreeNode> getLabels() {
            return labels;
        }

        public Builder<Spec, Status> setLabels(Map<String, YTreeNode> labels) {
            this.labels = new HashMap<>(labels);
            return this;
        }

        public Map<String, YTreeNode> getFields() {
            return fields;
        }

        public Builder<Spec, Status> setFields(Map<String, YTreeNode> fields) {
            this.fields = new HashMap<>(fields);
            return this;
        }

        public Builder<Spec, Status> addLabel(String key, YTreeNode value) {
            this.labels.put(key, value);
            return this;
        }

        public Builder<Spec, Status> addPrerequisite(YpAttributeTimestampPrerequisite prerequisite) {
            this.prerequisites.add(prerequisite);
            return this;
        }

        public Builder<Spec, Status> setRequestExtender(Consumer<YpObjectUpdate.Builder> requestExtender) {
            this.requestExtender = Optional.of(requestExtender);
            return this;
        }

        public UpdateYpObjectRequest<Spec, Status> build() {
            return new UpdateYpObjectRequest<>(spec, acl, status, labels, fields, prerequisites, requestExtender);
        }
    }

    public Optional<Spec> getSpec() {
        return spec;
    }

    public Optional<Acl> getAcl() {
        return acl;
    }

    public Optional<Status> getStatus() {
        return status;
    }

    public Map<String, YTreeNode> getLabels() {
        return labels;
    }

    public Map<String, YTreeNode> getFields() {
        return fields;
    }

    public List<YpAttributeTimestampPrerequisite> getPrerequisites() {
        return prerequisites;
    }

    public Optional<Consumer<YpObjectUpdate.Builder>> getRequestExtender() {
        return requestExtender;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof UpdateYpObjectRequest)) {
            return false;
        }
        UpdateYpObjectRequest<?, ?> that = (UpdateYpObjectRequest<?, ?>) o;
        return Objects.equals(spec, that.spec) &&
                Objects.equals(acl, that.acl) &&
                Objects.equals(status, that.status) &&
                Objects.equals(labels, that.labels) &&
                Objects.equals(fields, that.fields) &&
                Objects.equals(prerequisites, that.prerequisites);
    }

    @Override
    public int hashCode() {
        return Objects.hash(spec, acl, status, labels, prerequisites);
    }


    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("spec", spec)
                .add("acl", acl)
                .add("status", status)
                .add("labels", labels)
                .add("fields", fields)
                .add("prerequisites", prerequisites)
                .toString();
    }
}
