package ru.yandex.infra.controller.yp;

import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import one.util.streamex.StreamEx;

import ru.yandex.yp.YpRawObjectService;
import ru.yandex.yp.model.YpEvent;
import ru.yandex.yp.model.YpObjectType;
import ru.yandex.yp.model.YpWatchObjectsStatement;

class WatchObjectsExecutor extends ObjectsPaginationExecutor<List<YpEvent>> {
    private final YpRawObjectService ypClient;
    private final Long startTimestamp;
    private final CompletableFuture<Long> endTimestampGetter;
    private final YpObjectType ypObjectType;
    private final Duration timeLimit;

    public WatchObjectsExecutor(YpRawObjectService ypClient, Long startTimestamp,
                                CompletableFuture<Long> endTimestampGetter, int defaultPageSize, YpObjectType objectType,
                                Duration timeLimit)
    {
        super(defaultPageSize, objectType);
        super.setGetNextFunction(this::getNextPage);
        super.setUpdateAccumulatorFunction(this::updateStoredEvents);
        this.ypClient = ypClient;
        this.startTimestamp = startTimestamp;
        this.endTimestampGetter = endTimestampGetter;
        this.ypObjectType = objectType;
        this.timeLimit = timeLimit;
    }

    CompletableFuture<WatchedObjects> run() {
        return endTimestampGetter
                .thenApply(timestamp -> {
                    Map<String, List<YpEvent>> events = getObjects(timestamp, Optional.empty());
                    return new WatchedObjects(events, timestamp);
                });
    }

    private CompletableFuture<PageResult> getNextPage(Long timestamp, Optional<String> continuationTokenOpt) {
        YpWatchObjectsStatement.Builder builder = continuationTokenOpt
                .map(continuationToken -> YpWatchObjectsStatement.builder(ypObjectType, continuationToken))
                .orElseGet(() -> YpWatchObjectsStatement.builder(ypObjectType, startTimestamp));

        builder.setEventCountLimit((long) super.getCurrentPageSize())
                .setTimestamp(timestamp)
                .setTimeLimit(timeLimit);

        return ypClient.watchObjects(builder.build())
                .thenApply(watchedObjects -> new PageResult(
                        watchedObjects.getEvents().stream()
                                .collect(Collectors.<YpEvent, String>groupingBy(ypEvent -> ypEvent.getId().getId())),
                        watchedObjects.getContinuationToken()));
    }

    private void updateStoredEvents(Map<String, List<YpEvent>> storedEvents, Map<String, List<YpEvent>> newEvents) {
        newEvents.keySet().stream()
                .filter(storedEvents::containsKey)
                .forEach(existingId -> {
                    List<YpEvent> events = storedEvents.get(existingId);
                    events.addAll(newEvents.get(existingId));
                    storedEvents.put(existingId, events);
                });

        storedEvents.putAll(
                StreamEx.of(newEvents.keySet())
                        .filterBy(storedEvents::containsKey, false)
                        .toMap(newEvents::get));
    }
}
