package ru.yandex.infra.controller.yp;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import com.google.common.base.MoreObjects;

import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

// Fields are non-optional because consumer should know which fields he has requested and may expect them to be filled.
public class YpObject<Meta, Spec, Status> {
    private final Spec spec;
    private final Status status;
    private final Meta meta;
    private final Map<String, String> annotations;
    private final Map<String, YTreeNode> labels;
    private final long specTimestamp;
    private final long statusTimestamp;
    private final long metaTimestamp;

    public YpObject(Spec spec, Status status, Meta meta, Map<String, String> annotations,
            Map<String, YTreeNode> labels, long specTimestamp, long statusTimestamp, long metaTimestamp)
    {
        this.spec = spec;
        this.status = status;
        this.meta = meta;
        this.annotations = annotations;
        this.labels = labels;
        this.specTimestamp = specTimestamp;
        this.statusTimestamp = statusTimestamp;
        this.metaTimestamp = metaTimestamp;
    }

    public Spec getSpec() {
        return spec;
    }

    public Status getStatus() {
        return status;
    }

    public Meta getMeta() {
        return meta;
    }

    public Map<String, String> getAnnotations() {
        return annotations;
    }

    public Optional<String> getAnnotation(String key) {
        return Optional.ofNullable(annotations.get(key));
    }

    public Map<String, YTreeNode> getLabels() {
        return labels;
    }

    public Optional<YTreeNode> getLabel(String key) {
        return Optional.ofNullable(labels.get(key));
    }

    public long getSpecTimestamp() {
        return specTimestamp;
    }

    public long getStatusTimestamp() {
        return statusTimestamp;
    }

    public long getMetaTimestamp() {
        return metaTimestamp;
    }

    public static class Builder<Meta, Spec, Status> {
        private Spec spec;
        private Status status;
        private Meta meta;
        private Map<String, String> annotations = new HashMap<>();
        private Map<String, YTreeNode> labels = new HashMap<>();
        private long specTimestamp;
        private long metaTimestamp;
        private long statusTimestamp;

        public Spec getSpec() {
            return spec;
        }

        public Builder<Meta, Spec, Status> setSpec(Spec spec) {
            this.spec = spec;
            return this;
        }

        public Builder<Meta, Spec, Status> setSpecTimestamp(long timestamp) {
            this.specTimestamp = timestamp;
            return this;
        }

        public Builder<Meta, Spec, Status> setSpecAndTimestamp(Spec spec, long timestamp) {
            return setSpec(spec)
                    .setSpecTimestamp(timestamp);
        }

        public Status getStatus() {
            return status;
        }

        public Builder<Meta, Spec, Status> setStatus(Status status) {
            this.status = status;
            return this;
        }

        public Builder<Meta, Spec, Status> setStatusTimestamp(long timestamp) {
            this.statusTimestamp = timestamp;
            return this;
        }

        public Meta getMeta() {
            return meta;
        }

        public Builder<Meta, Spec, Status> setMeta(Meta meta) {
            this.meta = meta;
            return this;
        }

        public Builder<Meta, Spec, Status> setMetaTimestamp(long metaTimestamp) {
            this.metaTimestamp = metaTimestamp;
            return this;
        }

        public Map<String, String> getAnnotations() {
            return annotations;
        }

        public Builder<Meta, Spec, Status> setAnnotations(Map<String, String> annotations) {
            this.annotations = annotations;
            return this;
        }

        public Builder<Meta, Spec, Status> putAnnotation(String key, String value) {
            this.annotations.put(key, value);
            return this;
        }

        public Builder<Meta, Spec, Status> setLabels(Map<String, YTreeNode> labels) {
            this.labels = labels;
            return this;
        }

        public Builder<Meta, Spec, Status> putLabel(String key, YTreeNode value) {
            this.labels.put(key, value);
            return this;
        }

        public YpObject<Meta, Spec, Status> build() {
            return new YpObject<>(spec, status, meta, annotations, labels, specTimestamp, statusTimestamp, metaTimestamp);
        }
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof YpObject)) {
            return false;
        }
        YpObject<?, ?, ?> that = (YpObject<?, ?, ?>) o;
        return specTimestamp == that.specTimestamp &&
                statusTimestamp == that.statusTimestamp &&
                metaTimestamp == that.metaTimestamp &&
                Objects.equals(spec, that.spec) &&
                Objects.equals(status, that.status) &&
                Objects.equals(meta, that.meta) &&
                Objects.equals(annotations, that.annotations) &&
                Objects.equals(labels, that.labels);
    }

    @Override
    public int hashCode() {
        return Objects.hash(spec, status, meta, annotations, labels, specTimestamp, statusTimestamp, metaTimestamp);
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("spec", spec)
                .add("status", status)
                .add("meta", meta)
                .add("annotations", annotations)
                .add("labels", labels)
                .add("specTimestamp", specTimestamp)
                .add("statusTimestamp", statusTimestamp)
                .add("metaTimestamp", metaTimestamp)
                .toString();
    }
}
