package ru.yandex.infra.controller.yp;

import java.time.Duration;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.stream.Collectors;

import com.google.common.base.MoreObjects;
import com.typesafe.config.Config;

import ru.yandex.yp.model.YpObjectType;

public class YpObjectSettings {

    public final static String CONFIG_PARAMETER_USE_WATCHES = "use_watches";
    public final static String CONFIG_PARAMETER_WATCHES_TIMEOUT = "watches_timeout";
    public final static String CONFIG_PARAMETER_GET_OBJECTS_BATCH_SIZE = "get_objects_batch_size";
    public final static String CONFIG_PARAMETER_FULL_RELOAD_INTERVAL = "full_reload_interval";

    public final static YpObjectType TYPE_USED_AS_KEY_WITH_DEFAULT_SETTINGS = YpObjectType.NULL;

    private final boolean useWatches;
    private final Duration watchesTimeout;
    private final int getObjectsBatchSize;
    private final Duration fullReloadInterval;

    private YpObjectSettings(boolean useWatches,
                            int getObjectsBatchSize,
                            Duration fullReloadInterval,
                            Duration watchesTimeout) {
        this.useWatches = useWatches;
        this.getObjectsBatchSize = getObjectsBatchSize;
        this.fullReloadInterval = fullReloadInterval;
        this.watchesTimeout = watchesTimeout;
    }

    private YpObjectSettings(Config config, Config defaults) {
        this.useWatches = getValueFromConfigOrDefault(config,
                defaults,
                CONFIG_PARAMETER_USE_WATCHES,
                Config::getBoolean);
        this.watchesTimeout = getValueFromConfigOrDefault(config,
                defaults,
                CONFIG_PARAMETER_WATCHES_TIMEOUT,
                Config::getDuration);
        this.getObjectsBatchSize = getValueFromConfigOrDefault(config,
                defaults,
                CONFIG_PARAMETER_GET_OBJECTS_BATCH_SIZE,
                Config::getInt);
        this.fullReloadInterval = getValueFromConfigOrDefault(config,
                defaults,
                CONFIG_PARAMETER_FULL_RELOAD_INTERVAL,
                Config::getDuration);
    }

    private static <Value> Value getValueFromConfigOrDefault(Config config,
                                                             Config defaultConfig,
                                                             String parameterName,
                                                             BiFunction<Config, String, Value> extractFunction) {
        return config.hasPath(parameterName) ?
                extractFunction.apply(config, parameterName) :
                extractFunction.apply(defaultConfig, parameterName);
    }

    public static Map<YpObjectType, YpObjectSettings> loadFromConfig(Config config) {
        Config defaultConfig = config.getConfig("default");
        return config.root()
                .keySet()
                .stream()
                .collect(Collectors.toMap(
                        type -> type.equals("default") ?
                                TYPE_USED_AS_KEY_WITH_DEFAULT_SETTINGS :
                                YpObjectType.valueOf(type),
                        type -> new YpObjectSettings(config.getConfig(type), defaultConfig)
                ));
    }

    public static YpObjectSettings getSettingsForType(Map<YpObjectType, YpObjectSettings> allSettings, YpObjectType type) {
        return allSettings.getOrDefault(type, allSettings.get(TYPE_USED_AS_KEY_WITH_DEFAULT_SETTINGS));
    }

    public boolean isWatchesEnabled() {
        return useWatches;
    }

    public Duration getWatchesTimeout() {
        return watchesTimeout;
    }

    public int getGetObjectsBatchSize() {
        return getObjectsBatchSize;
    }

    public Duration getFullReloadInterval() {
        return fullReloadInterval;
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("useWatches", useWatches)
                .add("watchesTimeout", watchesTimeout)
                .add("getObjectsBatchSize", getObjectsBatchSize)
                .add("fullReloadInterval", fullReloadInterval)
                .toString();
    }

    public static class Builder {
        private boolean useWatches = false;
        private Duration watchesTimeout = Duration.ofSeconds(15);
        private int getObjectsBatchSize = 256;
        private Duration fullReloadInterval = Duration.ZERO;

        public Builder setWatches(boolean isEnabled) {
            this.useWatches = isEnabled;
            return this;
        }

        public Builder setWatchesTimeout(Duration watchesTimeout) {
            this.watchesTimeout = watchesTimeout;
            return this;
        }

        public Builder setFullReloadInterval(Duration fullReloadInterval) {
            this.fullReloadInterval = fullReloadInterval;
            return this;
        }

        public Builder setGetObjectsBatchSize(int getObjectsBatchSize) {
            this.getObjectsBatchSize = getObjectsBatchSize;
            return this;
        }

        public YpObjectSettings build() {
            return new YpObjectSettings(useWatches, getObjectsBatchSize, fullReloadInterval, watchesTimeout);
        }
    }
}
