package ru.yandex.infra.controller.yp;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.yp.YpPayloadDeserializer;
import ru.yandex.yp.YpRawObjectService;
import ru.yandex.yp.model.YpSelectStatement;

public class YpRequestWithPaging<T> {
    private static final Logger LOG = LoggerFactory.getLogger(YpRequestWithPaging.class);

    private final YpRawObjectService ypClient;
    private final int defaultPageSize;
    private final YpSelectStatement.Builder selectStatementBuilder;
    private final YpPayloadDeserializer<T> deserializer;
    private final List<T> result;
    private final CompletableFuture<List<T>> requestCompleted = new CompletableFuture<>();

    private YpRequestWithPaging(YpRawObjectService ypClient, int defaultPageSize,
                                YpSelectStatement.Builder selectStatementBuilder,
                                YpPayloadDeserializer<T> deserializer) {
        this.ypClient = ypClient;
        this.defaultPageSize = defaultPageSize;
        this.selectStatementBuilder = selectStatementBuilder;
        this.deserializer = deserializer;

        this.result = new ArrayList<>();
    }

    public static <T> CompletableFuture<List<T>> selectObjects(YpRawObjectService ypClient,
                                                               int defaultPageSize,
                                                               YpSelectStatement.Builder selectStatementBuilder,
                                                               YpPayloadDeserializer<T> deserializer) {
        var requestWithPaging = new YpRequestWithPaging<>(ypClient, defaultPageSize, selectStatementBuilder, deserializer);
        requestWithPaging.executeNextYpRequest(defaultPageSize);
        return requestWithPaging.requestCompleted;
    }

    private void executeNextYpRequest(int pageSize) {

        selectStatementBuilder.setLimit(pageSize);
        final YpSelectStatement statement = selectStatementBuilder.build();

        ypClient.selectObjects(statement, deserializer)
                .thenAccept(selectedObjects -> {

                    List<T> currentBatchItems = selectedObjects.getResults();
                    result.addAll(currentBatchItems);
                    Optional<String> continuationToken = selectedObjects.getContinuationToken();
                    if (currentBatchItems.size() < pageSize) {
                        requestCompleted.complete(result);
                    } else {
                        selectStatementBuilder.setContinuationToken(continuationToken.orElseThrow());
                        executeNextYpRequest(Math.min(defaultPageSize, pageSize * 2));
                    }

                })
                .exceptionally(error -> {

                    if (pageSize > 1 &&
                            error.getCause() instanceof StatusRuntimeException &&
                            ((StatusRuntimeException)error.getCause()).getStatus().getCode() == Status.Code.RESOURCE_EXHAUSTED) {

                        LOG.warn("Listing of {} with page size {} failed, will try reduced page size: {}",
                                statement.getObjectType(), pageSize, error.getMessage());
                        executeNextYpRequest(pageSize / 2);
                    } else {
                        requestCompleted.completeExceptionally(new ObjectsPaginationExcecutorException(error));
                    }

                    return null;
                });
    }
}
