package ru.yandex.infra.controller.yp;

import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

import ru.yandex.yp.YpRawObjectService;
import ru.yandex.yp.model.YpTransaction;

public class YpTransactionClientImpl implements YpTransactionClient {

    private final YpRawObjectService ypClient;

    public YpTransactionClientImpl(YpRawObjectService ypClient) {
        this.ypClient = ypClient;
    }

    public YpRawObjectService getYpRawObjectService() {
        return ypClient;
    }

    public CompletableFuture<YpTransaction> startTransaction() {
        return ypClient.startTransaction();
    }

    public CompletableFuture<?> commitTransaction(YpTransaction transaction) {
        return ypClient.commitTransaction(transaction);
    }

    public CompletableFuture<?> runWithTransaction(Function<YpTransaction, CompletableFuture<?>> function) {
        return commitTransactionWithErrorProcessing(
                startTransaction()
                        .thenCompose(transaction -> function.apply(transaction)
                                .thenApply(x -> transaction)));
    }

    private CompletableFuture<?> commitTransactionWithErrorProcessing(CompletableFuture<YpTransaction> completableTransaction) {
        //TODO(DEPLOY-1923): tests for transactions behaviour
        return completableTransaction
                .thenCompose(transaction -> ypClient.commitTransaction(transaction)
                        .whenComplete((x, errorReceived) -> {
                            if (errorReceived != null) {
                                ypClient.abortTransaction(transaction).whenComplete((resultAbort, errorAborting) -> {
                                    if (errorAborting != null) {
                                        throw new RuntimeException(String.format(
                                                "Receive error when abort transaction %s",
                                                transaction.toString()),
                                                errorAborting);
                                    }
                                });
                            }
        }));
    }

    public CompletableFuture<?> abortTransaction(YpTransaction transaction) {
        return ypClient.abortTransaction(transaction);
    }

}
