#include "leading_invader.h"

#include <mapreduce/yt/interface/client.h>

#include <library/cpp/threading/future/async.h>

#include <util/random/random.h>
#include <util/system/condvar.h>
#include <util/system/hostname.h>

namespace NInfra {
namespace NLeadingInvader {

namespace {

void ValidateConfig(const TConfig& config) {
    Y_ENSURE(FromString<TDuration>(config.GetTimeout()) > TDuration::Zero());
    Y_ENSURE(FromString<TDuration>(config.GetRetryAcquireLockInterval()) > TDuration::Zero());
}

class TLeadingInvader
    : public ILeadingInvader
{
public:
    TLeadingInvader(
        const TConfig& config
        , const std::function<void()>& onLockAcquired
        , const std::function<void()>& onLockLost
    )
        : Config_(config)
        , YtLockMode_(YtLockMode(Config_.GetLockMode()))
        , OnLockAqcuired_(onLockAcquired)
        , OnLockLost_(onLockLost)
    {
        ValidateConfig(config);

        AtomicSet(Running_, 1);
        Result_ = TError{TString("Acquire loop not started")};

        auto loop = [this] {
            while (AtomicGet(Running_) == 1) {
                try {
                    auto client = CreateClient();
                    {
                        TGuard<TMutex> guard(Mutex_);
                        Result_ = TError{TString("starting transaction")};
                    }
                    auto transaction = StartTransaction(client);
                    {
                        TGuard<TMutex> guard(Mutex_);
                        Result_ = TError{TString("creating lock")};
                    }
                    auto lock = transaction->Lock(Config_.GetPath(), YtLockMode_);
                    {
                        TGuard<TMutex> guard(Mutex_);
                        Result_ = TError{TString("waiting for lock")};
                    }
                    lock->GetAcquiredFuture().Wait();

                    {
                        TGuard<TMutex> guard(Mutex_);
                        Result_ = TExpected<void, TError>::DefaultSuccess();

                        OnLockAqcuired_();
                    }

                    while (AtomicGet(Running_) == 1) {
                        transaction->Ping();

                        TGuard<TMutex> guard(Mutex_);
                        if (AtomicGet(Running_) == 1) {
                            CondVar_.WaitT(Mutex_, FromString<TDuration>(Config_.GetTimeout()) / 4);
                        }
                    }

                    transaction->Abort();
                } catch (...) {
                    TGuard<TMutex> guard(Mutex_);
                    Result_ = TError{CurrentExceptionMessage()};
                }

                OnLockLost_();

                TGuard<TMutex> guard(Mutex_);
                if (AtomicGet(Running_) == 1) {
                    auto interval = FromString<TDuration>(Config_.GetRetryAcquireLockInterval());
                    CondVar_.WaitT(Mutex_, interval + interval * RandomNumber<float>());
                }
            }
        };

        ThreadPool_.Start(1);
        NThreading::Async(std::move(loop), ThreadPool_);
    }

    TExpected<void, TError> EnsureLeading() override {
        TGuard<TMutex> guard(Mutex_);
        return Result_;
    }

    TLeaderInfo GetLeaderInfo() const override {
        auto client = CreateClient();

        const NYT::TGetOptions getOptions = NYT::TGetOptions()
            .AttributeFilter(
                NYT::TAttributeFilter()
                    .AddAttribute("locks")
            );

        const NYT::TNode instanceNode = client->Get(Config_.GetPath(), getOptions);

        TString transactionId;
        for (const auto& lock : instanceNode.GetAttributes().AsMap().at("locks").AsList()) {
            if (FromString<NYT::ELockMode>(lock["mode"].AsString()) != NYT::ELockMode::LM_EXCLUSIVE) {
                continue;
            }

            if (lock["state"] != "acquired") {
                continue;
            }

            transactionId = lock["transaction_id"].AsString();
            break;
        }

        if (transactionId.empty()) {
            return TLeaderInfo{TLeaderInfo::EResolveLeaderStatus::NO_LEADER, "", ""};
        }

        const NYT::TGetOptions getOptionsForTransaction = NYT::TGetOptions()
            .AttributeFilter(
                NYT::TAttributeFilter()
                    .AddAttribute("hostname")
            );
        auto transaction = client->Get("//sys/transactions/" + transactionId, getOptionsForTransaction);

        try {
            const TString Fqdn = transaction.GetAttributes().AsMap().at("hostname").AsString();
            return TLeaderInfo{TLeaderInfo::EResolveLeaderStatus::SUCCEED, Fqdn, ""};
        } catch (...) {
            return TLeaderInfo{TLeaderInfo::EResolveLeaderStatus::FAILED, "", ""};
        }
    }

    static NYT::ELockMode YtLockMode(TConfig::ELockMode lockMode) {
        switch (lockMode) {
            case TConfig::EXCLUSIVE:
                return NYT::ELockMode::LM_EXCLUSIVE;
            case TConfig::SHARED:
                return NYT::ELockMode::LM_SNAPSHOT;
            default:
                ythrow yexception() << "Unknown lock mode \"" << TConfig::ELockMode_Name(lockMode) << "\"";
        }
    }

    ~TLeadingInvader() {
        {
            TGuard<TMutex> guard(Mutex_);
            AtomicSet(Running_, 0);
            CondVar_.Signal();
        }

        ThreadPool_.Stop();
    }

private:
    NYT::IClientPtr CreateClient() const {
        if (!Config_.GetToken().empty()) {
            auto result = NYT::CreateClient(
                Config_.GetProxy(),
                NYT::TCreateClientOptions().Token(Config_.GetToken()));

            return result;

        } else {
            return NYT::CreateClient(Config_.GetProxy());
        }
    }

    NYT::ITransactionPtr StartTransaction(NYT::IClientPtr& client) {
        NYT::TStartTransactionOptions opts;
        opts.Timeout(FromString<TDuration>(Config_.GetTimeout()));
        opts.AutoPingable(false);

        NYT::TNode attrs;
        attrs["hostname"] = HostName();
        opts.Attributes(attrs);

        return client->StartTransaction(opts);
    }

    TConfig Config_;
    NYT::ELockMode YtLockMode_;

    TExpected<void, TError> Result_;

    TAtomic Running_;
    TCondVar CondVar_;
    TThreadPool ThreadPool_;
    TMutex Mutex_;

    const std::function<void()> OnLockAqcuired_;
    const std::function<void()> OnLockLost_;
};

class TLeadingInvaderMock
    : public ILeadingInvader
{
public:
    TLeadingInvaderMock(
        bool isLeader
        , const std::function<void()>& onLockAcquired
    )
        : IsLeader_(isLeader)
    {
        if (IsLeader_) {
            onLockAcquired();
        }
    }

    TExpected<void, TError> EnsureLeading() override {
        if (IsLeader_) {
            return TExpected<void, TError>::DefaultSuccess();
        }
        return TError{TString("TLeadingInvaderMock: not a leader")};
    }

    TLeaderInfo GetLeaderInfo() const override {
        if (IsLeader_) {
            return TLeaderInfo{TLeaderInfo::EResolveLeaderStatus::SUCCEED, HostName(), ""};
        }
        return TLeaderInfo{TLeaderInfo::EResolveLeaderStatus::NO_LEADER, "", ""};
    }

private:
    bool IsLeader_;
};

} // namespace

TLeadingInvaderHolder CreateLeadingInvader(
    const TConfig& config
    , const std::function<void()>& onLockAcquired
    , const std::function<void()>& onLockLost
) {
    switch (config.GetType()) {
        case TConfig::YT_LOCK:
            return MakeHolder<TLeadingInvader>(config, onLockAcquired, onLockLost);
        case TConfig::MOCK_LEADER:
            return MakeHolder<TLeadingInvaderMock>(true, onLockAcquired);
        case TConfig::MOCK_NOT_LEADER:
            return MakeHolder<TLeadingInvaderMock>(false, onLockAcquired);
    }
}

} // namespace NLeadingInvader
} // namespace NInfra
