#include "lifetime_histogram.h"

#include <util/string/cast.h>

namespace NInfra::NLifetimeHistogram {

TResponseTimeHistogramsHolder::TResponseTimeHistogramsHolder(const TSensorGroup& sensorGroup, const TVector<TResponseTimeHistogramOptions>& responseTimeHistograms) {
    for (const auto& [operationType, bucketBounds, responseSizeTypes] : responseTimeHistograms) {
        auto& histograms = ResponseTimeHistograms_[operationType];
        auto& responseSizeBounds = ResponseSizeBounds_[operationType];
        responseSizeBounds.reserve(responseSizeTypes.size());
        for (const auto& [responseSizeType, responseSizeBound] : responseSizeTypes) {
            TString responseSizeTypeStr = ToString(responseSizeType);
            if (!histograms.contains(responseSizeTypeStr)) {
                histograms.emplace(
                    responseSizeTypeStr,
                    MakeIntrusive<NInfra::THistogramRateSensor>(
                        sensorGroup,
                        RESPONSE_TIME,
                        NMonitoring::ExplicitHistogram(bucketBounds),
                        TVector<std::pair<TStringBuf, TStringBuf>>{{OPERATION_TYPE, operationType}, {RESPONSE_SIZE, responseSizeTypeStr}}
                    )
                );
            }
            responseSizeBounds.emplace_back(std::move(responseSizeTypeStr), responseSizeBound);
        }
    }
}

TLifetimeHistogram TResponseTimeHistogramsHolder::GetLifetimeHistogram(TStringBuf operationType, const ui64& responseSize) const {
    const auto& histograms = ResponseTimeHistograms_.at(operationType);
    const TVector<std::pair<TString, ui64>>& responseSizeBounds = ResponseSizeBounds_.at(operationType);
    std::function<TIntrusivePtr<NInfra::THistogramRateSensor>()> getHistogram = [&histograms, &responseSizeBounds, &responseSize](){
        for (size_t i = 0; i + 1 < responseSizeBounds.size(); ++i) {
            if (responseSize < responseSizeBounds[i].second) {
                return histograms.at(responseSizeBounds[i].first);
            }
        }
        return histograms.at(responseSizeBounds.back().first);
    };
    return TLifetimeHistogram(std::move(getHistogram));
}

} // namespace NInfra::NLifetimeHistogram
