#include "event_iterator.h"
#include "hasher.h"

#include <library/cpp/eventlog/events_extension.h>
#include <library/cpp/streams/growing_file_input/growing_file_input.h>

#include <util/generic/buffer.h>
#include <util/string/split.h>
#include <util/string/type.h>
#include <util/system/unaligned_mem.h>

namespace NInfra {

const TEvent* IEventIterator::SafeNext() {
    while (true) {
        try {
            return Next();
        } catch (...) {
            Cerr << "Error occured: " << CurrentExceptionMessage() << Endl;
        }
    }
}

const TEvent* IEventIterator::SafePrev() {
    while (true) {
        try {
            return Prev();
        } catch (...) {
            Cerr << "Error occured: " << CurrentExceptionMessage() << Endl;
        }
    }
}

TEventIterator::TEventIterator(const TIteratorOptions& options)
    : StartTime_(options.StartTime)
    , EndTime_(options.EndTime)
    , File_(options.FileName, EOpenModeFlag::RdOnly)
    , MaxLogPriority_(options.MaxLogPriority)
    , EventAllowedPredicate_(options.EventPredicate)
{

    if (options.TailFMode) {
        Input_.Reset(new TGrowingFileInput(File_));
    } else {
        Input_.Reset(new TUnbufferedFileInput(File_));
    }

    FilePos_ = File_.GetPosition();

    if (options.EvList.empty()) {
        EventFilter_.Reset(nullptr);
    } else {
        EventFilter_.Reset(new TSet<ui64>());
        IncludeExcludeFlag_ = options.IncludeExcludeFlag;

        for (const auto& event : options.EvList) {
            if (IsNumber(event)) {
                EventFilter_->insert(FromString<ui64>(event));
            } else {
                EventFilter_->insert(NProtoBuf::TEventFactory::Instance()->IdByName(event));
            }
        }
    }
    if (options.FrameList.empty()) {
        FrameFilter_.Reset(nullptr);
    } else {
        FrameFilter_.Reset(new TSet<ui64>());
        FrameFilter_->insert(options.FrameList.begin(), options.FrameList.end());
    }

    if (!options.TailFMode) {
        JumpToStartTime();
    }
}

void TEventIterator::GoFromEndToAtLeastNEvents(size_t cnt) {
    i64 curPos = File_.GetLength();
    i64 firstSuccesReadEnd = std::numeric_limits<i64>::max();
    size_t numberOfEvents = 0;

    i64 stepBackFactor = 1;
    while (numberOfEvents < cnt + 1 && curPos > 0) {
        curPos = Max<i64>(curPos - TIteratorOptions::STEP_BACK_SIZE_BASE * stepBackFactor, 0);
        i64 firstReadNow = firstSuccesReadEnd;
        SetFilePosition(curPos);

        bool foundAtLeastOneEvent = false;
        while (SafeParse() && Event_ && FilePos_ < firstSuccesReadEnd) {
            foundAtLeastOneEvent = true;
            if (firstReadNow > FilePos_) {
                firstReadNow = FilePos_;
            }
            if (EventAllowed(Event_.Get())) {
                ++numberOfEvents;
            }
        }

        if (!foundAtLeastOneEvent) {
            if (TIteratorOptions::STEP_BACK_SIZE_BASE * stepBackFactor <= std::numeric_limits<i64>::max() / 2) {
                stepBackFactor *= 2;
            }
        } else {
            stepBackFactor = 1;
        }

        firstSuccesReadEnd = firstReadNow;
    }

    Skipped_ = 0;
    if (curPos == 0) {
        SetFilePosition(0);
    } else {
        SetFilePosition(firstSuccesReadEnd); // it's end of cnt + 1 or more event from end
    }
    Event_.Reset(nullptr);
}

void TEventIterator::JumpToStartTime() {
    if (StartTime_ == TIteratorOptions::MIN_START_TIME) {
        Skipped_ = 0;
        SetFilePosition(0);
        Event_.Reset(nullptr);
        return;
    }

    i64 minPos = -1;
    i64 maxPos = File_.GetLength();

    static constexpr i64 MAX_ALLOWED_SKIPPED_BYTES = 1 << 20;
    static constexpr ui8 MAX_LEFT_STEPS_TO_CONTINUE_ANYWAY = 10;
    while (maxPos - minPos > 1) {
        i64 curPos = minPos + (maxPos - minPos) / 2;
        SetFilePosition(curPos);
        if (SafeParse() && Event_ && StartTime_ > Event_->Timestamp() + TIteratorOptions::MAX_OVERLAP) {
            minPos = curPos;
        } else {
            maxPos = curPos;
        }

        // If we have large events, then SafeParse() may take too much time.
        // In this case sequential reading is better than binary search.
        if (Skipped_ > MAX_ALLOWED_SKIPPED_BYTES && CeilLog2(maxPos - minPos) > MAX_LEFT_STEPS_TO_CONTINUE_ANYWAY) {
            maxPos = minPos + 1;
            break;
        }

        Skipped_ = 0;
    }

    Skipped_ = 0;
    SetFilePosition(maxPos);
    Event_.Reset(nullptr);
}

bool TEventIterator::ParseMessage(NProtoBuf::Message* message, ui8 messageParsedHash, size_t size) {
    if (size == 0) {
        return true; // empty (or default) protomessage
    }

    TBuffer buffer;

    if (!ReadAndAppend(buffer, size) || GetOneByteHash(buffer.data(), size) != messageParsedHash) {
        return false;
    }

    return  message->ParseFromArray(buffer.data(), size);
}

bool TEventIterator::GetNextProtoMessage(NProtoBuf::Message* message, bool canSkip, size_t maxMessageSize) {
    size_t messageSize;
    ui8 messageHash;

    if (canSkip) {
        Skipped_ = 0;
    }

    for (;;) {
        i64 messageSizePos = FilePos_;
        if (ParseNumber(messageSize)) {
            if (messageSize <= maxMessageSize && ParseNumber(messageHash) &&
                ParseMessage(message, messageHash, messageSize))
            {
                return true;
            } else {
                if (canSkip) {
                    // unable to parse message, continue parsing messageSize from next byte
                    SetFilePosition(messageSizePos + 1);
                    ++Skipped_;
                } else {
                    return false;
                }
            }
        } else {
            return false;
        }
    }
}

bool TEventIterator::Parse() {
    bool success = SafeParse();
    ThrowAnExceptionIfWas();
    return success;
}

bool TEventIterator::SafeParse() {
    NLogEvent::TEventHeader header;
    TSimpleSharedPtr<NProtoBuf::Message> event;

    i64 tryCount = 0;

    for (;;) {
        i64 initialPos = FilePos_;
        if (!GetNextProtoMessage(&header, true, TIteratorOptions::MAX_HEADER_SIZE)) {
            Event_.Reset(nullptr);
            // number(less than sizeof(size_t)) of bytes left
            Skipped_ += File_.GetLength() - FilePos_;
            return false;
        }

        event.Reset(NProtoBuf::TEventFactory::Instance()->CreateEvent(header.GetEventId()));
        if (!event || !(GetNextProtoMessage(event.Get(), false))) {
            ++tryCount;
            SetFilePosition(initialPos + 1);
        } else {
            Event_.Reset(new TEvent(header, event));
            break;
        }
    }

    Skipped_ += tryCount;

    return true;
}

const TEvent* TEventIterator::Next() {
    if (WasException_) {
        WasException_ = false;
        if (EventAllowed(Event_.Get())) {
            return Event_.Get();
        }
    }

    while (Parse() && Event_->Timestamp() <= EndTime_ + TIteratorOptions::MAX_OVERLAP) { // events are not sorted by timestamp
        if (EventAllowed(Event_.Get())) {
            return Event_.Get();
        }
    }

    return nullptr;
}

bool TEventIterator::ReadAndAppend(TBuffer& buf, size_t len) {
    while (len > 0) {
        size_t readNow;
        size_t oldSize = buf.size();
        buf.Resize(oldSize + Min(MAX_BUFFER_SIZE, len));

        if (!(readNow = Input_->Read(buf.data() + oldSize, buf.size() - oldSize))) { // input exhausted
            return false;
        }

        len -= readNow;
        FilePos_ += readNow;
    }

    return true;
}

bool TEventIterator::EventAllowed(const TEvent* event) {
    return event
        && event->Timestamp() >= StartTime_
        && event->Timestamp() <= EndTime_
        && (!EventFilter_ || IncludeExcludeFlag_ == EventFilter_->contains(event->Id()))
        && (!FrameFilter_ || FrameFilter_->contains(event->FrameId()))
        && (!EventAllowedPredicate_ || EventAllowedPredicate_(event))
        && event->LogLevel() <= MaxLogPriority_;
    ;
}

void TEventIterator::SetFilePosition(i64 pos) {
    File_.Seek(pos, SeekDir::sSet);
    FilePos_ = pos;
}

void TEventIterator::ThrowAnExceptionIfWas() {
    if (Skipped_ > 0) {
        WasException_ = true;
        ythrow yexception() << "Parsing error. Bytes skipped: " << Skipped_;
    }
    WasException_ = false;
}

const TEvent* TCachedEventIterator::Next() {
    EventPositions_.push(FilePos_);
    const TEvent* event = TEventIterator::Next();
    if (!event) {
        EventPositions_.pop();
    }

    return event;
}

const TEvent* TCachedEventIterator::Prev() {
    if (!EventPositions_.empty()){
        EventPositions_.pop();
    }
    if (EventPositions_.empty()) {
        SetFilePosition(0);
        return nullptr;
    }

    SetFilePosition(EventPositions_.top());
    return TEventIterator::Next();
}

} // namespace NInfra
