#pragma once

#include "eventlog.h"

#include <util/generic/buffer.h>
#include <util/generic/ptr.h>
#include <util/generic/set.h>
#include <util/generic/string.h>
#include <util/generic/stack.h>
#include <util/generic/vector.h>
#include <util/stream/file.h>
#include <util/system/unaligned_mem.h>

namespace NInfra {

class IEventIterator {
public:
    virtual const TEvent* Next() {
        ythrow yexception() << "Not implemented";
    }

    virtual const TEvent* Prev() {
        ythrow yexception() << "Not implemented";
    }

    const TEvent* SafeNext();

    const TEvent* SafePrev();

    virtual ~IEventIterator() {};
};

using TEventIteratorPtr = TSimpleSharedPtr<IEventIterator>;

struct TIteratorOptions {
    TIteratorOptions& SetFileName(const TString& path) {
        FileName = path;

        return *this;
    }

    TIteratorOptions& SetStartTime(ui64 startTime) {
        StartTime = startTime;

        return *this;
    }

    TIteratorOptions& SetEndTime(ui64 endTime) {
        EndTime = endTime;

        return *this;
    }

    TIteratorOptions& SetEventList(const TVector<TString>& eventList) {
        EvList = eventList;

        return *this;
    }

    TIteratorOptions& SetFrameList(const TVector<ui64>& frameList) {
        FrameList = frameList;

        return *this;
    }

    TIteratorOptions& SetMaxLogPriority(ELogPriority maxLogPriority) {
        MaxLogPriority = maxLogPriority;

        return *this;
    }

    TIteratorOptions& SetEventPredicate(const std::function<bool(const TEvent*)>& predicate) {
        EventPredicate = predicate;

        return *this;
    }

public:
    static const ui64 MAX_OVERLAP = 30000000; // 30 seconds
    static const ui64 MIN_START_TIME = MAX_OVERLAP;
    static const ui64 MAX_END_TIME = ((ui64)-1) - MAX_OVERLAP;
    static const size_t MAX_HEADER_SIZE = 100; // roughly
    static const i64 STEP_BACK_SIZE_BASE = (1 << 10);

public:
    ui64 StartTime = MIN_START_TIME;
    ui64 EndTime = MAX_END_TIME;
    TString FileName = "";
    bool TailFMode = false;
    bool IncludeExcludeFlag = true;
    TVector<TString> EvList;
    TVector<ui64> FrameList;
    ELogPriority MaxLogPriority = LOG_MAX_PRIORITY;
    std::function<bool(const TEvent*)> EventPredicate;
};

class TEventIterator : public IEventIterator {
public:
    TEventIterator(const TIteratorOptions& options);

    const TEvent* Next() override;

    void GoFromEndToAtLeastNEvents(size_t cnt);

private:
    bool ParseMessage(NProtoBuf::Message* message, ui8 hash, size_t size);

    bool GetNextProtoMessage(NProtoBuf::Message* message, bool canSkip, size_t maxMessageSize = ::Max<size_t>());

    bool SafeParse();

    bool Parse();

    bool ReadAndAppend(TBuffer& buf, size_t len);

    bool EventAllowed(const TEvent* event);

    // throws an exception
    void ThrowAnExceptionIfWas();

    void JumpToStartTime();

protected:
    void SetFilePosition(i64 pos);

private:
    template <typename TValue>
    bool ParseNumber(TValue& input) {
        TBuffer buffer;

        if (!ReadAndAppend(buffer, sizeof(input))) {
            return false;
        }

        input = ReadUnaligned<TValue>(buffer.data());
        return true;
    }

private:
    THolder<IInputStream> Input_;
    THolder<TEvent> Event_;
    THolder<TSet<ui64>> EventFilter_;
    THolder<TSet<ui64>> FrameFilter_;
    ui64 StartTime_;
    ui64 EndTime_;
    bool IncludeExcludeFlag_;
    TFile File_;
    i64 Skipped_;
    bool WasException_ = false;
    ELogPriority MaxLogPriority_;
    const std::function<bool(const TEvent*)> EventAllowedPredicate_;

protected:
    i64 FilePos_;

private:
    static constexpr size_t MAX_BUFFER_SIZE = 1 << 20;
};

class TCachedEventIterator: public TEventIterator {
public:
    explicit TCachedEventIterator(const TIteratorOptions& options)
        : TEventIterator(options) {}

    const TEvent* Next() override;

    const TEvent* Prev() override;

private:
    TStack<i64> EventPositions_;
};

} // namespace NInfra
