#include "eventlog.h"

#include "hasher.h"

#include <library/cpp/colorizer/colors.h>
#include <library/cpp/json/json_prettifier.h>
#include <library/cpp/protobuf/json/proto2json.h>

#include <util/system/unaligned_mem.h>
#include <util/datetime/base.h>

namespace NInfra {

void TEvent::PrintHeader(const NLogEvent::TEventHeader& header, IOutputStream& out, const TEventContext& context) {
    NColorizer::TColors colors;
    if (context.NoTTy) {
        colors.SetIsTTY(false);
    }

    if (context.HumanReadable) {
        out << TInstant::MicroSeconds(header.GetTimestamp()).ToIsoStringLocal() << '\t';
    } else {
        out << header.GetTimestamp() << '\t';
    }

    if (context.PrintTimestampDiff) {
        i64 delta = static_cast<i64>(header.GetTimestamp()) - context.PrevEventTime;
        out << (delta >= 0 ? "+" : "") << delta << "us\t";
    }

    if (context.PrintFrameId) {
        out << header.GetFrameId() << '\t';
    }

    if (ELogPriority(header.GetLogLevel()) <= ELogPriority::TLOG_ERR) {
        out << colors.RedColor();
    } else if (ELogPriority(header.GetLogLevel()) <= ELogPriority::TLOG_NOTICE) {
        out << colors.YellowColor();
    } else {
        out << colors.GreenColor();
    }

    out << ELogPriority(header.GetLogLevel()) << '\t';
    out << colors.OldColor();

    if (context.PrintSourceLocation && header.HasSourceFile()) {
        out << TSourceLocation(header.GetSourceFile(), header.GetSourceLine()) << '\t';
    }
}

void TEvent::PrintAsJson(const NLogEvent::TEventHeader& header, const NProtoBuf::Message& message, IOutputStream& out, const TEventContext& context) {
    PrintHeader(header, out, context);

    NColorizer::TColors colors;
    if (context.NoTTy) {
        colors.SetIsTTY(false);
    }
    out
        << colors.BoldColor()
        << NProtoBuf::TEventFactory::Instance()->NameById(header.GetEventId())
        << colors.OldColor()
        << (context.Pretty ? '\n' : '\t');

    NProtobufJson::TProto2JsonConfig config;
    config.SetEnumMode(NProtobufJson::TProto2JsonConfig::EnumName);
    if (context.Pretty) {
        config
            .SetFormatOutput(true)
            .SetMapAsObject(true);

        const TString jsonMessage = NProtobufJson::Proto2Json(message, config);
        NJson::PrettifyJson(jsonMessage, out, /* unquote */ true, /* padding */ 2);
    } else {
        NProtobufJson::Proto2Json(message, out, config);
    }
}

void TEvent::PrintProtoToBuffer(const NProtoBuf::Message& message, TBuffer& buffer) {
    size_t messageSize = message.ByteSizeLong();
    buffer.Resize(buffer.Size() + sizeof(size_t) + sizeof(ui8) + messageSize);

    WriteUnaligned<size_t>(buffer.Pos() - messageSize - sizeof(ui8) - sizeof(size_t), messageSize);

    message.SerializeWithCachedSizesToArray((unsigned char*)buffer.Pos() - messageSize);

    ui8 hash = GetOneByteHash(buffer.Pos() - messageSize, messageSize);
    WriteUnaligned<ui8>(buffer.Pos() - messageSize - sizeof(ui8), hash);
}

} // NInfra
