#pragma once

#include <infra/libs/logger/protos/header.pb.h>

#include <google/protobuf/text_format.h>

#include <library/cpp/eventlog/events_extension.h>
#include <library/cpp/logger/priority.h>

#include <util/generic/buffer.h>
#include <util/generic/maybe.h>

namespace NInfra {
class TEvent {
public:
    using TEventTimestamp = ui64;

    struct TEventContext {
        bool HumanReadable = false;
        bool PrintTimestampDiff = false;
        bool PrintFrameId = false;
        TEventTimestamp PrevEventTime = 0;
        bool NoTTy = false;
        bool Pretty = false;
        bool PrintSourceLocation = true;
    };

public:
    static void PrintHeader(const NLogEvent::TEventHeader& header, IOutputStream& out, const TEventContext& context);
    static void PrintAsJson(const NLogEvent::TEventHeader& header, const NProtoBuf::Message& message, IOutputStream& out, const TEventContext& context);

    static void PrintProtoToBuffer(const NProtoBuf::Message& message, TBuffer& out);

    TEvent(const NLogEvent::TEventHeader& header, TSimpleSharedPtr<NProtoBuf::Message> message)
        : Header_(header)
        , Message_(message)
    {
    }

    ui64 Id() const {
        return Header_.GetEventId();
    }

    ui64 LogLevel() const {
        return Header_.GetLogLevel();
    }

    ui64 FrameId() const {
        return Header_.GetFrameId();
    }

    ui64 Timestamp() const {
        return Header_.GetTimestamp();
    }

    TMaybe<TSourceLocation> SourceLocation() const {
        return TSourceLocation(Header_.GetSourceFile(), Header_.GetSourceLine());
    }

    TString GetData() const {
        TString output;
        google::protobuf::TextFormat::PrintToString(*Message_, &output);

        return output;
    }

    NProtoBuf::Message* GetProto() const {
        return Message_.Get();
    }

    template <typename T>
    const T* Get() const {
        return static_cast<const T*>(this->GetProto());
    }

    void PrintAsJson(IOutputStream& out, const TEventContext& context) const {
        PrintAsJson(Header_, *Message_, out, context);
    }

    void PrintHeader(IOutputStream& out, const TEventContext& context) const {
        PrintHeader(Header_, out, context);
    }

private:
    const NLogEvent::TEventHeader Header_;
    TSimpleSharedPtr<NProtoBuf::Message> Message_;
};

} // namespace NInfra
