#pragma once

#include <infra/libs/logger/eventlog.h>
#include <infra/libs/logger/protos/config.pb.h>
#include <infra/libs/logger/protos/events.ev.pb.h>
#include <infra/libs/logger/protos/header.pb.h>

#include <infra/libs/error_booster_client/error_booster_client.h>

#include <library/cpp/logger/log.h>
#include <library/cpp/threading/future/async.h>

#include <util/generic/buffer.h>
#include <util/stream/buffer.h>
#include <util/system/unaligned_mem.h>

namespace NInfra {

////////////////////////////////////////////////////////////////////////////////

class TLogFrame;

using TLogFramePtr = TAtomicSharedPtr<TLogFrame>;

class TLogFrame {
public:
    TLogFrame(
        TAtomicSharedPtr<TLog> logger
        , TAtomicSharedPtr<NErrorBoosterClient::TErrorBoosterClient> errorBoosterClient
        , TAtomic id
        , TAtomicSharedPtr<TAtomic> level
        , TLoggerConfig::ELogFormat format
    )
        : Logger_(logger)
        , ErrorBoosterClient_(errorBoosterClient)
        , FrameId_(id)
        , Level_(level)
        , Format_(format)
    {
    }

    TLogFrame(TLogFrame&& frame)
        : Level_(frame.Level_)
        , Format_(frame.Format_)
    {
        DoSwap(FrameId_, frame.FrameId_);
        DoSwap(Logger_, frame.Logger_);
    }

    bool AcceptLevel(ELogPriority level) const {
        return static_cast<ui8>(level) <= static_cast<ui8>(AtomicGet(*Level_));
    }

    void LogEvent(
        ELogPriority level,
        const TMaybe<TSourceLocation>& sourceLocation,
        const NProtoBuf::Message& event,
        ui32 eventId
    ) {
        if (AcceptLevel(level)) {
            NLogEvent::TEventHeader header(CreateHeader(eventId, level, sourceLocation));

            if (ErrorBoosterClient_) {
                ErrorBoosterClient_->LogEvent(header, event);
            }
            TBuffer buffer;
            TEvent::PrintProtoToBuffer(header, buffer);
            TEvent::PrintProtoToBuffer(event, buffer);
            WriteData(level, buffer.data(), buffer.size());
        }
    }

    void LogEvent(ELogPriority level, const NProtoBuf::Message& event, ui32 eventId) {
        LogEvent(level, Nothing(), event, eventId);
    }

    template <typename T>
    void LogEvent(
        ELogPriority level,
        const TMaybe<TSourceLocation>& sourceLocation,
        const T& event
    ) {
        if (AcceptLevel(level)) {
            NLogEvent::TEventHeader header(CreateHeader(event.ID, level, sourceLocation));

            if (ErrorBoosterClient_) {
                ErrorBoosterClient_->LogEvent(header, event);
            }
            TBuffer buffer;
            if (Format_ == TLoggerConfig_ELogFormat_PROTO) {
                TEvent::PrintProtoToBuffer(header, buffer);
                TEvent::PrintProtoToBuffer(event, buffer);
            } else {
                TBufferOutput out(buffer);
                TEvent::PrintAsJson(header, event, out, TEvent::TEventContext{.HumanReadable = false, .PrintFrameId = true});
                out << Endl;
            }
            WriteData(level, buffer.data(), buffer.size());
        }
    }

    template <typename T>
    void LogEvent(ELogPriority level, const T& event) {
        LogEvent(level, Nothing(), event);
    }

    void LogEvent(const NProtoBuf::Message& event, ui32 eventId) {
        LogEvent(ELogPriority::TLOG_INFO, Nothing(), event, eventId);
    }

    template <typename TEvent>
    void LogEvent(const TEvent& event) {
        LogEvent(ELogPriority::TLOG_INFO, Nothing(), event);
    }

private:
    NLogEvent::TEventHeader CreateHeader(const ui32 eventId, ELogPriority level, const TMaybe<TSourceLocation>& sourceLocation) {
        NLogEvent::TEventHeader protoHeader;
        protoHeader.SetTimestamp(Now().MicroSeconds());
        protoHeader.SetFrameId(FrameId_);
        protoHeader.SetLogLevel(static_cast<ui32>(level));
        protoHeader.SetEventId(eventId);
        if (const TSourceLocation* sourceLocationData = sourceLocation.Get()) {
            protoHeader.SetSourceFile(sourceLocationData->File.data(), sourceLocationData->File.size());
            protoHeader.SetSourceLine(sourceLocationData->Line);
        }
        return protoHeader;
    }

private:
    void WriteData(ELogPriority level, const char* data, size_t len) {
        Logger_->Write(level, data, len);
    }

private:
    TAtomicSharedPtr<TLog> Logger_;
    TAtomicSharedPtr<NErrorBoosterClient::TErrorBoosterClient> ErrorBoosterClient_;
    TAtomic FrameId_;
    const TAtomicSharedPtr<TAtomic> Level_;
    const TLoggerConfig::ELogFormat Format_;
};

////////////////////////////////////////////////////////////////////////////////

#define FRAME_LOG_EVENT(frame, level, ...) \
    do { \
        if (frame->AcceptLevel(level)) { \
            frame->LogEvent(level, __LOCATION__, __VA_ARGS__); \
        } \
    } while (false)


#define INFRA_LOG_DEBUG(...) FRAME_LOG_EVENT(logFrame, ELogPriority::TLOG_DEBUG, __VA_ARGS__)
#define INFRA_LOG_INFO(...) FRAME_LOG_EVENT(logFrame, ELogPriority::TLOG_INFO, __VA_ARGS__)
#define INFRA_LOG_WARN(...) FRAME_LOG_EVENT(logFrame, ELogPriority::TLOG_WARNING, __VA_ARGS__)
#define INFRA_LOG_ERROR(...) FRAME_LOG_EVENT(logFrame, ELogPriority::TLOG_ERR, __VA_ARGS__)

////////////////////////////////////////////////////////////////////////////////

} // namespace NInfra
