#include "log_parser.h"
#include "hasher.h"

#include <library/cpp/eventlog/events_extension.h>

namespace {

constexpr size_t MAX_BUFFER_SIZE = 1 << 20;

bool ReadAndAppend(IInputStream& input, TBuffer& buf, size_t len) {
    while (len > 0) {
        size_t readNow;
        size_t oldSize = buf.size();
        buf.Resize(oldSize + Min(MAX_BUFFER_SIZE, len));

        if (!(readNow = input.Read(buf.data() + oldSize, buf.size() - oldSize))) { // input exhausted
            return false;
        }

        len -= readNow;
    }
    return true;
}

template <typename TValue>
bool ParseNumber(IInputStream& input, TValue& val) {
    TBuffer buffer;

    if (!ReadAndAppend(input, buffer, sizeof(val))) {
        return false;
    }

    val = ReadUnaligned<TValue>(buffer.data());
    return true;
}

bool ParseMessage(IInputStream& input, NProtoBuf::Message* message, ui8 messageParsedHash, size_t size) {
    if (size == 0) {
        return true; // empty (or default) protomessage
    }

    TBuffer buffer;

    if (!ReadAndAppend(input, buffer, size) || NInfra::GetOneByteHash(buffer.data(), size) != messageParsedHash) {
        return false;
    }

    return  message->ParseFromArray(buffer.data(), size);
}

bool ParseProtoMessage(IInputStream& input, NProtoBuf::Message* message) {
    size_t messageSize;
    ui8 messageHash;
    for (;;) {
        if (ParseNumber(input, messageSize)) {
            return ParseNumber(input, messageHash) && ParseMessage(input, message, messageHash, messageSize);
        }
    }
}

} // namespace

namespace NInfra {

TEvent ParseEvent(IInputStream& input) {
    TEvent event;

    ParseProtoMessage(input, &event.Header);
    if (auto eventPtr = NProtoBuf::TEventFactory::Instance()->CreateEvent(event.Header.GetEventId()); eventPtr) {
        event.Event.Reset(eventPtr);
        ParseProtoMessage(input, event.Event.Get());
    } else {
        ythrow yexception() << "Event " << event.Header.GetEventId() << " isn't registered. Probably, it should be added to infra/libs/logger/all_events?";
    }

    return event;
}

} // namespace NInfra
