#include "log_printer.h"

#include <contrib/libs/re2/re2/re2.h>

#include <library/cpp/build_info/build_info.h>
#include <library/cpp/eventlog/dumper/common.h>
#include <library/cpp/eventlog/dumper/evlogdump.h>
#include <library/cpp/getopt/small/last_getopt.h>
#include <library/cpp/svnversion/svnversion.h>

#include <util/datetime/base.h>
#include <util/generic/serialized_enum.h>

namespace NInfra {

namespace NPrivate {

void PrintVersionAndExit() {
    Cout << GetProgramSvnVersion() << Endl;
    Cout << GetBuildInfo() << Endl;

    exit(0);
}

void PrintEvents(
    const TIteratorOptions &options
    , bool humanReadable
    , bool printTimestampDiff
    , bool printFrameId
    , bool colorlessOutput
    , bool prettyPrint
    , bool printSourceLocation
) {
    TEventIterator iter(options);

    TEvent::TEventContext context{
        .HumanReadable = humanReadable
        , .PrintTimestampDiff = printTimestampDiff
        , .PrintFrameId = printFrameId
        , .NoTTy = colorlessOutput
        , .Pretty = prettyPrint
        , .PrintSourceLocation = printSourceLocation
    };
    while (const TEvent* event = iter.SafeNext()) {
        if (context.PrevEventTime == 0) {
            context.PrevEventTime = event->Timestamp();
        }
        event->PrintAsJson(Cout, context);
        Cout << Endl;
        context.PrevEventTime = event->Timestamp();
    }
}

void PrintLastNEvents(
    const TIteratorOptions &options
    , size_t cnt
    , IOutputStream& out
    , bool humanReadable
    , bool printTimestampDiff
    , bool printFrameId
    , bool colorlessOutput
    , bool prettyPrint
    , bool printSourceLocation
) {
    TEventIterator iter(options);
    TDeque<TEvent> events;

    iter.GoFromEndToAtLeastNEvents(cnt);
    while (const TEvent* event = iter.SafeNext()) {
        events.push_back(*event);
        if (events.size() > cnt) {
            events.pop_front();
        }
    }

    TEvent::TEventContext context{
        .HumanReadable = humanReadable
        , .PrintTimestampDiff = printTimestampDiff
        , .PrintFrameId = printFrameId
        , .NoTTy = colorlessOutput
        , .Pretty = prettyPrint
        , .PrintSourceLocation = printSourceLocation
    };
    for (const auto& event : events) {
        if (context.PrevEventTime == 0) {
            context.PrevEventTime = event.Timestamp();
        }
        event.PrintAsJson(out, context);
        out << Endl;
        context.PrevEventTime = event.Timestamp();
    }
}

void PrintEventLogErrorsAndExit(const TString& filePath) {
    TIteratorOptions options;
    TEventIterator iter(options.SetFileName(filePath));
    while (const TEvent* event = iter.SafeNext());

    exit(0);
}

} // namespace NPrivate

int PrintEventLog(int argc, const char* argv[]) {
    NLastGetopt::TOpts opts;

    opts.SetFreeArgsNum(0, 1);
    opts.SetFreeArgTitle(0, "<eventlog>", "Event log file");

    TString start;
    opts.AddLongOption(
            's', "start-time",
            "Start time (Unix time in microseconds, ISO8601, or HH:MM[:SS] (MSK) in the last 24 hours)")
        .Optional()
        .StoreResult(&start);

    TString end;
    opts.AddLongOption(
            'e', "end-time",
            "End time (Unix time in microseconds, ISO8601, or HH:MM[:SS] (MSK) in the last 24 hours)")
        .Optional()
        .StoreResult(&end);

    TIteratorOptions options;

    TVector<TString> includeEventList;
    opts.AddLongOption(
            'i', "event-list",
            "Comma-separated list of included event class IDs or names "
            "(e.g. 265,287 or MainTaskError,ContextCreated)")
        .Optional()
        .SplitHandler(&includeEventList, ',')
        .StoreValue(&options.IncludeExcludeFlag, true);

    TVector<TString> excludeEventList;
    opts.AddLongOption(
            'x', "exclude-list",
            "Comma-separated list of excluded event class IDs or names (see also --event-list option)")
        .Optional()
        .SplitHandler(&excludeEventList, ',')
        .StoreValue(&options.IncludeExcludeFlag, false);

    TVector<ui64> frameList;
    opts.AddLongOption(
            'm', "frame-id",
            "Select by frame id list (e.g. 1,8,245)")
        .Optional()
        .SplitHandler(&frameList, ',');

    opts.AddLongOption(
            'v', "version",
            "Print program version")
        .NoArgument()
        .Optional()
        .Handler(&NPrivate::PrintVersionAndExit);

    bool printErrors = false;
    opts.AddLongOption(
            'p', "print-errors",
            "Print eventlog errors (like missed bytes) and exit, ignoring all print-options (-x, -t ...)")
        .NoArgument()
        .Optional()
        .StoreValue(&printErrors, true);

    opts.AddLongOption(
            't', "tail",
            "Open file like `tail -f` does")
        .NoArgument()
        .Optional()
        .StoreValue(&options.TailFMode, true);

    size_t cnt = std::numeric_limits<size_t>::max();
    opts.AddLongOption(
            'n', "number",
            "Print the last n log events")
        .Optional()
        .StoreResult(&cnt);

    bool humanReadable = false;
    opts.AddLongOption(
            'r', "human-readable",
            "Print some fields (e.g. timestamp) in human-readable format, add time offsets")
        .NoArgument()
        .StoreValue(&humanReadable, true);

    bool printTimestampDiff = false;
    opts.AddLongOption(
            'd', "print-timestamp-diff",
            "Print diff between events' timestamps")
        .NoArgument()
        .StoreValue(&printTimestampDiff, true);

    bool printFrameId = false;
    opts.AddLongOption(
            'f', "print-frame-id",
            "Print frame id")
        .NoArgument()
        .StoreValue(&printFrameId, true);

    bool colorlessOutput = false;
    opts.AddLongOption(
            'c', "colorless-output",
             "Print colorless logs")
        .NoArgument()
        .StoreValue(&colorlessOutput, true);

    opts.AddLongOption(
            'l', "logs-level",
            TStringBuilder() << "Maximal log level priority (" << GetEnumAllNames<ELogPriority>() << ")")
        .Optional()
        .StoreResult(&options.MaxLogPriority)
        .DefaultValue(LOG_MAX_PRIORITY);

    bool prettyPrint = false;
    opts.AddLongOption(
             "pretty",
             "Print events in formatted JSON"
        )
        .NoArgument()
        .StoreValue(&prettyPrint, true);

    bool printSourceLocation = true;
    opts.AddLongOption(
             "no-source-location",
             "Do not print source location"
        )
        .NoArgument()
        .StoreValue(&printSourceLocation, false);

    TString helpEvents;
    opts.AddLongOption("help-events")
        .RequiredArgument("event name, event id or string \"all\"")
        .Optional()
        .StoreResult(&helpEvents);

    NLastGetopt::TOptsParseResult parsedOpts(&opts, argc, argv);

    if (helpEvents) {
        PrintHelpEvents(helpEvents, NEvClass::Factory());
        return EXIT_SUCCESS;
    }

    if (parsedOpts.GetFreeArgs().size() == 0) {
        parsedOpts.PrintUsage();
        exit(1);
    }

    if (printErrors) {
        NPrivate::PrintEventLogErrorsAndExit(parsedOpts.GetFreeArgs()[0]);
    }

    if (includeEventList && excludeEventList) {
        throw NLastGetopt::TUsageException() << "You can use either include (-i) or exclude (-x) events list. ";
    }

    options.SetFileName(parsedOpts.GetFreeArgs()[0])
        .SetStartTime(ParseTime(start, TIteratorOptions::MIN_START_TIME))
        .SetEndTime(ParseTime(end, TIteratorOptions::MAX_END_TIME))
        .SetEventList(options.IncludeExcludeFlag ? includeEventList : excludeEventList)
        .SetFrameList(frameList);

    if (cnt != std::numeric_limits<size_t>::max()) {
        NPrivate::PrintLastNEvents(
            options
            , cnt
            , Cout
            , humanReadable
            , printTimestampDiff
            , printFrameId
            , colorlessOutput
            , prettyPrint
            , printSourceLocation
        );
    } else {
        NPrivate::PrintEvents(
            options
            , humanReadable
            , printTimestampDiff
            , printFrameId
            , colorlessOutput
            , prettyPrint
            , printSourceLocation
        );
    }

    return EXIT_SUCCESS;
}

} // namespace NInfra
